<?php

declare(strict_types=1);

namespace Drupal\drupalfit\Service;

use Drupal\drupalfit\FitCheckGroupPluginManager;
use Drupal\drupalfit\FitCheckPluginManager;
use Drupal\drupalfit\FitResultCollection;
use Drupal\drupalfit\FitScoreCalculator;

/**
 * Collects and generates fit reports by running fit checks.
 */
final class FitReportCollector implements FitReportCollectorInterface {

  /**
   * Cache of check definitions.
   *
   * @var array
   */
  private array $checkDefinitions = [];

  /**
   * Constructs a new FitReportCollector object.
   */
  public function __construct(
    private readonly FitCheckPluginManager $pluginManagerFitCheck,
    private readonly FitCheckGroupPluginManager $checkGroupPluginManager,
    private readonly DrupalFitApiClient $apiClient,
  ) {}

  /**
   * {@inheritDoc}
   */
  public function generate(): FitResultCollection {
    if (empty($this->checkDefinitions)) {
      $this->checkDefinitions = $this->pluginManagerFitCheck->getDefinitions();
    }

    $results = new FitResultCollection();
    foreach ($this->checkDefinitions as $checkDefinition) {
      try {
        /** @var \Drupal\drupalfit\FitCheckInterface $check */
        $check = $this->pluginManagerFitCheck
          ->createInstance($checkDefinition['id'], $checkDefinition);
        $result = $check->execute();
        if ($result !== NULL) {
          $results->add($result);
        }
      }
      catch (\Exception $e) {
        // Log error and continue with next check.
        continue;
      }
    }
    return $results;
  }

  /**
   * {@inheritDoc}
   */
  public function calculateScore(FitResultCollection $collection): FitScoreCalculator {
    $checkGroupDefinitions = $this->checkGroupPluginManager->getDefinitions();
    $weightConfiguration = [];
    $scoreWeightConfiguration = [];
    $externalProviderConfiguration = [];

    foreach ($checkGroupDefinitions as $checkGroupDefinition) {
      $groupId = $checkGroupDefinition['id'];

      if (isset($checkGroupDefinition['weight'])) {
        $weightConfiguration[$groupId] = $checkGroupDefinition['weight'];
      }

      if (isset($checkGroupDefinition['scoreWeight'])) {
        $scoreWeightConfiguration[$groupId] = $checkGroupDefinition['scoreWeight'];
      }

      if (isset($checkGroupDefinition['externalProvider'])) {
        $externalProviderConfiguration[$groupId] = $checkGroupDefinition['externalProvider'];
      }
    }

    $externalScores = $this->getExternalScores();

    return FitScoreCalculator::create($collection, $weightConfiguration, $scoreWeightConfiguration, $externalProviderConfiguration, $externalScores);
  }

  /**
   * Gets external scores from API.
   */
  private function getExternalScores(): array {
    $scores = $this->apiClient->getAuditScores();

    if (!$scores) {
      return [];
    }

    $externalScores = [];
    if ($scores->accessibility !== NULL) {
      $externalScores['accessibility'] = $scores->accessibility;
    }
    if ($scores->seo !== NULL) {
      $externalScores['seo'] = $scores->seo;
    }

    return $externalScores;
  }

}
