<?php

declare(strict_types=1);

namespace Drupal\drupalfit\Controller;

use League\CommonMark\MarkdownConverter;
use League\CommonMark\Extension\Table\TableExtension;
use League\CommonMark\Extension\CommonMark\CommonMarkCoreExtension;
use League\CommonMark\Environment\Environment;
use Drupal\Core\Extension\ExtensionList;
use Drupal\Core\Controller\ControllerBase;
use Drupal\drupalfit\DrupalFitConstants;
use Drupal\drupalfit\Service\FitReportCollectorInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Request;

/**
 * Controller for Drupal Fit security audit reports.
 */
final class DrupalFitReport extends ControllerBase {

  /**
   * Constructs a DrupalFitReport controller.
   */
  public function __construct(
    private readonly FitReportCollectorInterface $fitReportCollector,
    private readonly ExtensionList $extensionList,
  ) {
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): DrupalFitReport {
    return new DrupalFitReport(
      $container->get('drupalfit.report_collector'),
      $container->get('extension.list.module')
    );
  }

  /**
   * Displays the main Drupal Fit security audit report.
   */
  public function report(): array {
    $results = $this->fitReportCollector->generate();

    $scores = $this->fitReportCollector
      ->calculateScore($results)
      ->calculateScores();

    return [
      '#theme' => 'drupalfit_report',
      '#results' => $results,
      '#scores' => $scores,
      '#logo_url' => '/' . $this->extensionList->getPath('drupalfit') . '/assets/drupalfit.svg',
      '#cache' => [
        'max-age' => 0,
      ],
      '#attached' => [
        'library' => [
          'drupalfit/security-report',
        ],
      ],
    ];
  }

  /**
   * Displays the Drupalfit.com iframe embed tab.
   *
   * @return array
   *   Render array for the iframe embed or registration prompt.
   */
  public function drupalFitReport(Request $request): array {
    $config = $this->config('drupalfit.settings');
    $apiKey = $config->get('api_key');

    if (empty($apiKey)) {
      return [
        '#theme' => 'drupalfit_register',
        '#logo_path' => '/' . $this->extensionList->getPath('drupalfit') . '/assets/drupalfit.svg',
        '#register_url' => DrupalFitConstants::DRUPALFIT_BASE_URL . '/register',
        '#attached' => ['library' => ['drupalfit/security-report']],
      ];
    }

    $iframeUrl = DrupalFitConstants::DRUPALFIT_BASE_URL;
    return [
      '#theme' => 'drupalfit_embed',
      '#iframe_url' => $iframeUrl,
      '#iframe_title' => $this->t('Drupal Fit'),
      "#cache" => ['max-age' => 0],
      '#attached' => [
        'library' => ['drupalfit/security-report'],
        'drupalSettings' => [
          'drupalfit' => [
            'apiKey' => $apiKey,
            'domain' => $config->get('domain') ?: $request->getHost(),
          ],
        ],
      ],
    ];
  }

  /**
   * Displays the DrupalFit help and documentation.
   *
   * @return array
   *   Render array for the help page.
   */
  public function help(): array {
    $modulePath = $this->extensionList->getPath('drupalfit');
    $readmePath = $modulePath . '/README.md';

    if (!file_exists($readmePath)) {
      return ['#markup' => $this->t('Documentation not found.')];
    }

    $markdown = file_get_contents($readmePath);
    $config = [
      'table' => [
        'wrap' => [
          'enabled' => TRUE,
          'tag' => 'div',
          'attributes' => ['class' => 'table-responsive'],
        ],
      ],
    ];
    $environment = new Environment($config);
    $environment->addExtension(new CommonMarkCoreExtension());
    $environment->addExtension(new TableExtension());
    $converter = new MarkdownConverter($environment);
    $html = $converter->convert($markdown)->getContent();

    return [
      '#type' => 'container',
      '#attributes' => ['class' => ['drupalfit-help']],
      'content' => ['#markup' => $html],
      '#attached' => ['library' => ['drupalfit/security-report']],
    ];
  }

}
