<?php

declare(strict_types=1);

namespace Drupal\drupalfit\Plugin\FitCheck;

use Drupal\Core\Database\Connection;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\drupalfit\Attribute\FitCheck;
use Drupal\drupalfit\Enum\FitWeight;
use Drupal\drupalfit\FitCheckPluginBase;
use Drupal\drupalfit\FitResult;
use Drupal\drupalfit\Plugin\FitCheckGroup\PerformanceGroup;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation of the fit_check.
 */
#[FitCheck(
  id: 'anonymous_user_session',
  fitGroup: PerformanceGroup::GROUP_ID,
  label: new TranslatableMarkup('Anonymous User Sessions'),
  description: new TranslatableMarkup('Checks if anonymous sessions are being unnecessarily created.'),
  successMessage: new TranslatableMarkup('No anonymous sessions detected.'),
  failureMessage: new TranslatableMarkup('Anonymous sessions are being created.'),
)]
class AnonymousUserSession extends FitCheckPluginBase {

  /**
   * {@inheritDoc}
   */
  public function __construct(
    array $configuration,
    string $plugin_id,
    mixed $plugin_definition,
    protected readonly Connection $database,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * Inject the required config, and services.
   */
  public static function create(
    ContainerInterface $container,
    array $configuration,
    $plugin_id,
    mixed $plugin_definition,
  ): static {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('database')
    );
  }

  /**
   * {@inheritDoc}
   */
  public function execute(): FitResult {
    $result = FitResult::create(
      $this->getPluginId(),
      $this->label(),
      $this->fitGroup(),
      FitWeight::Ok,
      $this->description()
    );

    $anonSessionCount = $this->database->select('sessions', 's')
      ->condition('uid', 0)
      ->countQuery()
      ->execute()
      ->fetchField();

    if ($anonSessionCount > 0) {
      $result
        ->setWeight(FitWeight::Medium)
        ->setFailureMessage($this->t('Found @count anonymous session(s) in the database.', ['@count' => $anonSessionCount]))
        ->setHelpMessage($this->t('Anonymous sessions consume memory and database resources.'))
        ->setHelpMessage($this->t('Review modules that may set cookies or session data for anonymous users.'))
        ->setHelpMessage($this->t('Consider using the Session Limit module or reviewing custom code that calls session_start().'));
    }
    else {
      $result->setSuccessMessage($this->successMessage());
    }

    return $result;
  }

}
