<?php

declare(strict_types=1);

namespace Drupal\drupalfit\Plugin\FitCheck;

use Drupal\Core\Site\Settings;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\drupalfit\Attribute\FitCheck;
use Drupal\drupalfit\Enum\FitWeight;
use Drupal\drupalfit\FitCheckPluginBase;
use Drupal\drupalfit\FitResult;
use Drupal\drupalfit\Plugin\FitCheckGroup\ContentAndConfigGroup;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation of the fit_check.
 */
#[FitCheck(
  id: 'config_dir',
  fitGroup: ContentAndConfigGroup::GROUP_ID,
  label: new TranslatableMarkup('Config Sync Directory Location'),
  description: new TranslatableMarkup('Checks if config sync directory is outside web root.'),
  successMessage: new TranslatableMarkup('Config sync directory is outside web root.'),
  failureMessage: new TranslatableMarkup('Config sync directory is inside web root.'),
)]
class ConfigDir extends FitCheckPluginBase {

  /**
   * {@inheritDoc}
   */
  public function __construct(
    array $configuration,
    string $plugin_id,
    mixed $plugin_definition,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * Inject the required services.
   */
  public static function create(
    ContainerInterface $container,
    array $configuration,
    $plugin_id,
    mixed $plugin_definition,
  ): static {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
    );
  }

  /**
   * {@inheritDoc}
   */
  public function execute(): FitResult {
    $result = FitResult::create(
      $this->getPluginId(),
      $this->label(),
      $this->fitGroup(),
      FitWeight::Ok
    );

    $configDir = Settings::get('config_sync_directory');
    $webRoot = DRUPAL_ROOT;

    $realConfigPath = $configDir ? realpath($configDir) : FALSE;
    $realWebRoot = realpath($webRoot);

    if ($realConfigPath && $realWebRoot && str_starts_with($realConfigPath, $realWebRoot)) {
      $result
        ->setWeight(FitWeight::High)
        ->setFailureMessage($this->failureMessage())
        ->setHelpMessage([
          '#type' => 'inline_template',
          '#template' => '{{ text }}<br><strong>{{ label }}</strong> {{ path }}<br><br>{{ recommendation }}',
          '#context' => [
            'text' => $this->t('Config sync directory is accessible via web, exposing sensitive configuration data.'),
            'label' => $this->t('Current path:'),
            'path' => $configDir,
            'recommendation' => $this->t('Move config directory outside web root (e.g., ../config/sync) in settings.php.'),
          ],
        ]);
    }
    else {
      $result->setSuccessMessage($this->successMessage());
    }

    return $result;
  }

}
