<?php

declare(strict_types=1);

namespace Drupal\drupalfit\Plugin\FitCheck;

use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\drupalfit\Attribute\FitCheck;
use Drupal\drupalfit\Enum\FitWeight;
use Drupal\drupalfit\FitCheckPluginBase;
use Drupal\drupalfit\FitResult;
use Drupal\drupalfit\Plugin\FitCheckGroup\AccessibilityGroup;
use Drupal\drupalfit\Service\DrupalFitApiClient;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Plugin implementation of the fit_check.
 */
#[FitCheck(
 id: 'drupalfit_accessibility_check',
 fitGroup: AccessibilityGroup::GROUP_ID,
 label: new TranslatableMarkup('Accessibility Report'),
 description: new TranslatableMarkup('View detailed accessibility analysis from DrupalFit.'),
 successMessage: new TranslatableMarkup('View accessibility report.'),
 failureMessage: new TranslatableMarkup('Accessibility data available.'),
 )]
class DrupalFitAccessibility extends FitCheckPluginBase {

  /**
   * {@inheritDoc}
   */
  public function __construct(
    array $configuration,
    string $plugin_id,
    mixed $plugin_definition,
    protected RequestStack $requestStack,
    protected DrupalFitApiClient $apiClient,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * Inject the required config, and services.
   */
  public static function create(
    ContainerInterface $container,
    array $configuration,
    $plugin_id,
    mixed $plugin_definition,
  ): static {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('request_stack'),
      $container->get('drupalfit.api_client'),
    );
  }

  /**
   * {@inheritDoc}
   */
  public function execute(): ?FitResult {
    $scores = $this->apiClient->getAuditScores();
    if (!$scores || $scores->accessibility === NULL) {
      return NULL;
    }
    $result = FitResult::create(
      $this->getPluginId(),
      $this->label(),
      $this->fitGroup(),
      FitWeight::Info,
      $this->description()
    );
    $result->setInfoMessage($this->t('Accessibility Score: @score/100', ['@score' => $scores->accessibility]));
    return $result;
  }

}
