<?php

declare(strict_types=1);

namespace Drupal\drupalfit\Plugin\FitCheck;

use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\Update\UpdateRegistry;
use Drupal\drupalfit\Attribute\FitCheck;
use Drupal\drupalfit\Enum\FitWeight;
use Drupal\drupalfit\FitCheckPluginBase;
use Drupal\drupalfit\FitResult;
use Drupal\drupalfit\Plugin\FitCheckGroup\BestPracticesGroup;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation of the fit_check.
 */
#[FitCheck(
  id: 'pending_db_update',
  fitGroup: BestPracticesGroup::GROUP_ID,
  label: new TranslatableMarkup('Pending Database Updates'),
  description: new TranslatableMarkup('Checks for pending database updates.'),
  successMessage: new TranslatableMarkup('No pending database updates.'),
  failureMessage: new TranslatableMarkup('Pending database updates detected.'),
)]
class PendingDbUpdate extends FitCheckPluginBase {

  /**
   * {@inheritDoc}
   */
  public function __construct(
    array $configuration,
    string $plugin_id,
    mixed $plugin_definition,
    protected readonly UpdateRegistry $updateRegistry,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * Inject the required services.
   */
  public static function create(
    ContainerInterface $container,
    array $configuration,
    $plugin_id,
    mixed $plugin_definition,
  ): static {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('update.post_update_registry'),
    );
  }

  /**
   * {@inheritDoc}
   */
  public function execute(): FitResult {
    require_once DRUPAL_ROOT . '/core/includes/install.inc';
    drupal_load_updates();

    $result = FitResult::create(
      $this->getPluginId(),
      $this->label(),
      $this->fitGroup(),
      FitWeight::Ok
    );

    $pending = $this->getUpdateDbStatus();

    if (empty($pending)) {
      $result->setSuccessMessage($this->successMessage());
    }
    else {
      $items = [];
      foreach ($pending as $update) {
        $desc = $update['description'] ?? '';
        $items[] = '<strong>' . $update['module'] . '</strong> - ' . $update['update_id'] . $desc;
      }

      $result
        ->setWeight(FitWeight::High)
        ->setFailureMessage($this->failureMessage())
        ->setHelpMessage([
          '#type' => 'inline_template',
          '#template' => '{{ text }}<br><br>{{ items|raw }}<br><br><a href="{{ url }}">{{ link }}</a>',
          '#context' => [
            'text' => $this->t('Run pending database updates to ensure your site is up to date.'),
            'items' => implode('<br>', $items),
            'url' => '/update.php',
            'link' => $this->t('Run updates'),
          ],
        ]);
    }

    return $result;
  }

  /**
   * Returns information about available module updates.
   */
  protected function getUpdateDbStatus(): array {
    require_once DRUPAL_ROOT . '/core/includes/update.inc';
    $pending = \update_get_update_list();
    $return = [];

    foreach ($pending as $module => $updates) {
      if (isset($updates['start'])) {
        foreach ($updates['pending'] as $update_id => $description) {
          $description = str_replace($update_id . ' -   ', '', $description);
          $return[$module . "_update_$update_id"] = [
            'module' => $module,
            'update_id' => $update_id,
            'description' => $description,
            'type' => 'hook_update_n',
          ];
        }
      }
    }

    $postUpdates = $this->updateRegistry->getPendingUpdateInformation();
    foreach ($postUpdates as $module => $post_update) {
      if (isset($post_update['pending'])) {
        foreach ($post_update['pending'] as $id => $description) {
          $return[$module . '_post_' . $id] = [
            'module' => $module,
            'update_id' => $id,
            'description' => trim($description),
            'type' => 'post_update',
          ];
        }
      }
    }

    return $return;
  }

}
