<?php

declare(strict_types=1);

namespace Drupal\drupalfit\Plugin\FitCheck;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\drupalfit\Attribute\FitCheck;
use Drupal\drupalfit\Enum\FitWeight;
use Drupal\drupalfit\FitCheckPluginBase;
use Drupal\drupalfit\FitResult;
use Drupal\drupalfit\Plugin\FitCheckGroup\BestPracticesGroup;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation of the fit_check.
 */
#[FitCheck(
  id: 'proper_error_handling',
  fitGroup: BestPracticesGroup::GROUP_ID,
  label: new TranslatableMarkup('Proper Error Handling'),
  description: new TranslatableMarkup('Checks proper logging and error handling configuration.'),
  successMessage: new TranslatableMarkup('Logging and error handling properly configured.'),
  failureMessage: new TranslatableMarkup('Logging and error handling needs improvement.'),
)]
class ProperErrorHandling extends FitCheckPluginBase {

  /**
   * {@inheritDoc}
   */
  public function __construct(
    array $configuration,
    string $plugin_id,
    mixed $plugin_definition,
    protected readonly ConfigFactoryInterface $configFactory,
    protected readonly ModuleHandlerInterface $moduleHandler,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * Inject required configuration and services.
   */
  public static function create(
    ContainerInterface $container,
    array $configuration,
    $plugin_id,
    mixed $plugin_definition,
  ): static {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('config.factory'),
      $container->get('module_handler'),
    );
  }

  /**
   * {@inheritDoc}
   */
  public function execute(): FitResult {
    $result = FitResult::create(
      $this->getPluginId(),
      $this->label(),
      $this->fitGroup(),
      FitWeight::Ok
    );

    $issues = [];
    $isProd = $this->determineEnvironmentType() === 'Production';

    if ($isProd && ini_get('display_errors')) {
      $issues[] = $this->t('Display errors is enabled in production');
    }

    if ($isProd && ini_get('display_startup_errors')) {
      $issues[] = $this->t('Display startup errors is enabled in production');
    }

    $errorLevel = $this->configFactory->get('system.logging')->get('error_level');
    if ($isProd && in_array($errorLevel, ['all', 'verbose', 'some'])) {
      $issues[] = $this->t('Error level set to display errors in production');
    }

    if (!ini_get('log_errors')) {
      $issues[] = $this->t('PHP error logging is disabled');
    }

    $hasDblLog = $this->moduleHandler->moduleExists('dblog');
    $hasSyslog = $this->moduleHandler->moduleExists('syslog');

    if (!$hasDblLog && !$hasSyslog) {
      $issues[] = $this->t('No logging module enabled (dblog or syslog)');
    }

    if (empty($issues)) {
      $result->setSuccessMessage($this->successMessage());
    }
    else {
      $weight = $isProd && (ini_get('display_errors') || in_array($errorLevel, ['all', 'verbose'])) ? FitWeight::Critical : FitWeight::Medium;
      $result
        ->setWeight($weight)
        ->setFailureMessage($this->failureMessage())
        ->setHelpMessage([
          '#type' => 'inline_template',
          '#template' => '<strong>{{ label }}</strong><br>{{ items|raw }}<br><br><a href="{{ url }}">{{ link }}</a>',
          '#context' => [
            'label' => $this->t('Issues found:'),
            'items' => implode('<br>', $issues),
            'url' => '/admin/config/development/logging',
            'link' => $this->t('Configure logging'),
          ],
        ]);
    }

    return $result;
  }

  /**
   * Predictable env.
   *
   * @todo have find the better way.
   */
  private function determineEnvironmentType(): string {
    $env = $_ENV['ENVIRONMENT'] ?? $_SERVER['ENVIRONMENT'] ?? 'Unknown';
    return in_array(strtolower($env), ['prod', 'production']) ? 'Production' : 'Development';
  }

}
