<?php

declare(strict_types=1);

namespace Drupal\drupalfit\Plugin\FitCheck;

use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\drupalfit\Attribute\FitCheck;
use Drupal\drupalfit\Enum\FitWeight;
use Drupal\drupalfit\FitCheckPluginBase;
use Drupal\drupalfit\FitResult;
use Drupal\drupalfit\Plugin\FitCheckGroup\BestPracticesGroup;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation of the fit_check.
 */
#[FitCheck(
  id: 'syslog',
  fitGroup: BestPracticesGroup::GROUP_ID,
  label: new TranslatableMarkup('Syslog Module'),
  description: new TranslatableMarkup('Checks if syslog module is enabled for better logging.'),
  successMessage: new TranslatableMarkup('Syslog module is enabled.'),
  failureMessage: new TranslatableMarkup('Syslog module is not enabled.'),
)]
class SysLog extends FitCheckPluginBase {

  /**
   * {@inheritDoc}
   */
  public function __construct(
    array $configuration,
    string $plugin_id,
    mixed $plugin_definition,
    protected readonly ModuleHandlerInterface $moduleHandler,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * Inject the required services.
   */
  public static function create(
    ContainerInterface $container,
    array $configuration,
    $plugin_id,
    mixed $plugin_definition,
  ): static {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('module_handler'),
    );
  }

  /**
   * {@inheritDoc}
   */
  public function execute(): FitResult {
    $result = FitResult::create(
      $this->getPluginId(),
      $this->label(),
      $this->fitGroup(),
      FitWeight::Ok
    );

    if ($this->moduleHandler->moduleExists('dblog')) {
      $result->setSuccessMessage($this->successMessage());
    }
    else {
      $result
        ->setWeight(FitWeight::Low)
        ->setFailureMessage($this->failureMessage())
        ->setHelpMessage([
          '#type' => 'inline_template',
          '#template' => '{{ text }} <a href="{{ url }}">{{ link }}</a>',
          '#context' => [
            'text' => $this->t('Enable the Syslog module to send log messages to the system log for better performance and centralized logging.'),
            'url' => '/admin/modules',
            'link' => $this->t('Enable Syslog'),
          ],
        ]);
    }

    return $result;
  }

}
