<?php

declare(strict_types=1);

namespace Drupal\drupalfit\Plugin\FitCheck;

use Drupal\Core\Url;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\drupalfit\Attribute\FitCheck;
use Drupal\drupalfit\Enum\FitWeight;
use Drupal\drupalfit\FitCheckPluginBase;
use Drupal\drupalfit\FitResult;
use Drupal\drupalfit\Plugin\FitCheckGroup\SecurityGroup;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation of the fit_check.
 */
#[FitCheck(
  id: 'user_one_check',
  fitGroup: SecurityGroup::GROUP_ID,
  label: new TranslatableMarkup('User 1 Account'),
  description: new TranslatableMarkup('Checks if the admin account (UID 1) is blocked.'),
  successMessage: new TranslatableMarkup('User 1 is blocked.'),
  failureMessage: new TranslatableMarkup('User 1 is active.'),
)]
class UserOneAccountCheck extends FitCheckPluginBase {

  /**
   * {@inheritDoc}
   */
  public function __construct(
    array $configuration,
    string $plugin_id,
    mixed $plugin_definition,
    protected readonly EntityTypeManagerInterface $entityTypeManager,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * Inject the required config, and services.
   */
  public static function create(
    ContainerInterface $container,
    array $configuration,
    $plugin_id,
    mixed $plugin_definition,
  ): static {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_type.manager')
    );
  }

  /**
   * {@inheritDoc}
   */
  public function execute(): FitResult {
    /** @var \Drupal\user\UserInterface $adminAccount */
    $adminAccount = $this->entityTypeManager
      ->getStorage('user')
      ->load(1);

    $weight = FitWeight::Ok;

    $result = FitResult::create(
      $this->getPluginId(),
      $this->label(),
      $this->fitGroup(),
      $weight
    );

    if (!$adminAccount || $adminAccount->isBlocked()) {
      $result->setSuccessMessage($this->successMessage());
    }
    else {
      $result
        ->setWeight(FitWeight::Critical)
        ->setFailureMessage($this->failureMessage())
        ->setHelpMessage([
          '#type' => 'inline_template',
          '#template' => '{{ message }} {{ link }}',
          '#context' => [
            'message' => $this->t('Block user "@name" to prevent unauthorized access.', ['@name' => $adminAccount->getAccountName()]),
            'link' => [
              '#type' => 'link',
              '#title' => $this->t('Edit user 1'),
              '#url' => Url::fromRoute('entity.user.edit_form', ['user' => 1]),
            ],
          ],
        ]);
    }
    return $result;
  }

}
