<?php

declare(strict_types=1);

namespace Drupal\drupalfit\Plugin\FitCheck;

use Drupal\Core\Url;
use Drupal\Core\Database\Connection;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\drupalfit\Attribute\FitCheck;
use Drupal\drupalfit\Enum\FitWeight;
use Drupal\drupalfit\FitCheckPluginBase;
use Drupal\drupalfit\FitResult;
use Drupal\drupalfit\Plugin\FitCheckGroup\PerformanceGroup;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation of the fit_check.
 */
#[FitCheck(
  id: 'watchdog_table_size',
  fitGroup: PerformanceGroup::GROUP_ID,
  label: new TranslatableMarkup('Watchdog Table Size'),
  description: new TranslatableMarkup('Checks if the watchdog (dblog) table is too large.'),
  successMessage: new TranslatableMarkup('Watchdog table size is acceptable.'),
  failureMessage: new TranslatableMarkup('Watchdog table is too large.'),
)]
class WatchdogTableSize extends FitCheckPluginBase {

  /**
   * {@inheritDoc}
   */
  public function __construct(
    array $configuration,
    string $plugin_id,
    mixed $plugin_definition,
    protected readonly Connection $database,
    protected readonly ModuleHandlerInterface $moduleHandler,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * Inject the required services.
   */
  public static function create(
    ContainerInterface $container,
    array $configuration,
    $plugin_id,
    mixed $plugin_definition,
  ): static {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('database'),
      $container->get('module_handler'),
    );
  }

  /**
   * {@inheritDoc}
   */
  public function execute(): FitResult {
    $result = FitResult::create(
      $this->getPluginId(),
      $this->label(),
      $this->fitGroup(),
      FitWeight::Ok,
      $this->description()
    );

    if (!$this->moduleHandler->moduleExists('dblog')) {
      $result->setSuccessMessage($this->t('Database logging is disabled.'));
      return $result;
    }

    $rowCount = $this->database->select('watchdog', 'w')
      ->countQuery()
      ->execute()
      ->fetchField();

    $estimatedMB = round($rowCount * 2 / 1024, 2);

    if ($rowCount > 1000000) {
      $result
        ->setWeight(FitWeight::High)
        ->setFailureMessage($this->t(
          'Watchdog table has @count rows (~@size MB).',
          ['@count' => number_format($rowCount), '@size' => $estimatedMB]
        ))
        ->setHelpMessage($this->t('Large watchdog tables slow down queries and consume disk space.'))
        ->setHelpMessage([
          '#type' => 'link',
          '#title' => $this->t('Logging settings'),
          '#url' => Url::fromRoute('system.logging_settings'),
        ])
        ->setHelpMessage($this->t('Consider using Syslog module or external logging instead of dblog.'));
    }
    elseif ($rowCount > 100000) {
      $result
        ->setWeight(FitWeight::Medium)
        ->setFailureMessage($this->t(
          'Watchdog table has @count rows (~@size MB).',
          ['@count' => number_format($rowCount), '@size' => $estimatedMB]
        ))
        ->setHelpMessage([
          '#type' => 'link',
          '#title' => $this->t('Logging settings'),
          '#url' => Url::fromRoute('system.logging_settings'),
        ]);
    }
    else {
      $result->setSuccessMessage($this->successMessage());
    }

    return $result;
  }

}
