<?php

namespace Drupal\drupitor_client\Access;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Routing\Access\AccessInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Component\Routing\Route;

/**
 * Access check for Drupitor API endpoints.
 */
class DrupitorApiAccessCheck implements AccessInterface, ContainerInjectionInterface {

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The logger factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected $loggerFactory;

  /**
   * The request stack.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected $requestStack;

  /**
   * Constructs a DrupitorApiAccessCheck object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory.
   * @param \Symfony\Component\HttpFoundation\RequestStack $request_stack
   *   The request stack.
   */
  public function __construct(ConfigFactoryInterface $config_factory, LoggerChannelFactoryInterface $logger_factory, RequestStack $request_stack) {
    $this->configFactory = $config_factory;
    $this->loggerFactory = $logger_factory;
    $this->requestStack = $request_stack;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('config.factory'),
      $container->get('logger.factory'),
      $container->get('request_stack')
    );
  }

  /**
   * Checks access to the Drupitor API.
   *
   * @param \Symfony\Component\Routing\Route $route
   *   The route to check against.
   *
   * @return \Drupal\Core\Access\AccessResultInterface
   *   The access result.
   */
  public function access(Route $route = NULL): AccessResultInterface {
    $request = $this->requestStack->getCurrentRequest();
    $config = $this->configFactory->get('drupitor_client.settings');
    $logger = $this->loggerFactory->get('drupitor_client');

    // Check if the feature is enabled
    if (!$config->get('enabled')) {
      $logger->warning('Drupitor API endpoint accessed but feature is disabled. IP: @ip', [
        '@ip' => $request->getClientIp(),
      ]);
      return AccessResult::forbidden('Drupitor client feature is disabled');
    }

    // Get the configured API token
    $configured_token = $config->get('api_token');
    if (empty($configured_token)) {
      $logger->warning('Drupitor API endpoint accessed but no API token is configured. IP: @ip', [
        '@ip' => $request->getClientIp(),
      ]);
      return AccessResult::forbidden('API token not configured');
    }

    // Check for API token in request
    $provided_token = $this->getTokenFromRequest($request);
    if (empty($provided_token)) {
      $logger->warning('Drupitor API endpoint accessed without API token. IP: @ip', [
        '@ip' => $request->getClientIp(),
      ]);
      return AccessResult::forbidden('API token required');
    }

    // Use timing-safe comparison to prevent timing attacks
    if (!hash_equals($configured_token, $provided_token)) {
      $logger->warning('Drupitor API endpoint accessed with invalid API token. IP: @ip, Token: @token', [
        '@ip' => $request->getClientIp(),
        '@token' => substr($provided_token, 0, 8) . '...',
      ]);
      return AccessResult::forbidden('Invalid API token');
    }

    $logger->info('Drupitor API endpoint accessed successfully. IP: @ip', [
      '@ip' => $request->getClientIp(),
    ]);

    return AccessResult::allowed();
  }

  /**
   * Extracts the API token from the request.
   *
   * Supports multiple token formats:
   * - Authorization header: "Bearer TOKEN"
   * - Authorization header: "TOKEN"
   * - Query parameter: "token=TOKEN"
   * - Custom header: "X-API-Token: TOKEN"
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request object.
   *
   * @return string|null
   *   The API token or NULL if not found.
   */
  protected function getTokenFromRequest($request): ?string {
    // Check Authorization header (Bearer format)
    $auth_header = $request->headers->get('Authorization');
    if ($auth_header) {
      if (preg_match('/^Bearer\s+(.+)$/i', $auth_header, $matches)) {
        return trim($matches[1]);
      }
      // Also support direct token in Authorization header
      if (preg_match('/^[a-zA-Z0-9._-]+$/', trim($auth_header))) {
        return trim($auth_header);
      }
    }

    // Check custom X-API-Token header
    $api_token_header = $request->headers->get('X-API-Token');
    if ($api_token_header) {
      return trim($api_token_header);
    }

    // Check query parameter (less secure, but for compatibility)
    $query_token = $request->query->get('token');
    if ($query_token) {
      return trim($query_token);
    }

    return NULL;
  }

}