<?php

namespace Drupal\drupitor_client\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Response;

/**
 * Controller for Drupitor client.
 */
class DrupitorClientController extends ControllerBase {

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The logger factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected $loggerFactory;

  /**
   * The messenger service.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * Constructs a DrupitorClientController object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger service.
   */
  public function __construct(ConfigFactoryInterface $config_factory, LoggerChannelFactoryInterface $logger_factory, MessengerInterface $messenger) {
    $this->configFactory = $config_factory;
    $this->loggerFactory = $logger_factory;
    $this->messenger = $messenger;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('config.factory'),
      $container->get('logger.factory'),
      $container->get('messenger')
    );
  }

  /**
   * Gets all installed packages with version information.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   JSON response with package information.
   */
  public function getUpdates(): JsonResponse {
    $config = $this->configFactory->get('drupitor_client.settings');
    $logger = $this->loggerFactory->get('drupitor_client');

    // Check if the feature is enabled
    if (!$config->get('enabled')) {
      $logger->warning('Drupitor client endpoint accessed but feature is disabled.');
      return new JsonResponse([
        'error' => 'Drupitor client feature is disabled',
      ], Response::HTTP_FORBIDDEN);
    }

    try {
      $package_data = $this->collectUpdates();

      // Encrypt the data before sending
      $encrypted_data = $this->encryptData($package_data);

      $logger->info('Drupitor client package information retrieved and encrypted successfully.');

      return new JsonResponse([
        'status' => 'success',
        'timestamp' => time(),
        'encrypted' => true,
        'data' => $encrypted_data,
      ]);
    }
    catch (\Exception $e) {
      $logger->error('Error retrieving Drupitor client package information: @message', [
        '@message' => $e->getMessage(),
      ]);

      return new JsonResponse([
        'error' => 'Unable to retrieve package information',
        'timestamp' => time(),
      ], Response::HTTP_INTERNAL_SERVER_ERROR);
    }
  }

  /**
   * Collects all installed packages with their information.
   *
   * @return array
   *   Array of all installed packages with version information.
   */
  protected function collectUpdates(): array {
    $project_root = $this->getProjectRoot();

    if (!$project_root) {
      throw new \Exception('Unable to determine project root directory.');
    }

    $composer_json_path = $project_root . '/composer.json';
    $composer_lock_path = $project_root . '/composer.lock';

    // Validate file paths and existence
    if (!$this->isValidPath($composer_json_path) || !file_exists($composer_json_path)) {
      throw new \Exception('composer.json not found or invalid path.');
    }

    if (!$this->isValidPath($composer_lock_path) || !file_exists($composer_lock_path)) {
      throw new \Exception('composer.lock not found or invalid path.');
    }

    // Parse composer files safely
    $composer_json = $this->parseJsonFile($composer_json_path);
    $composer_lock = $this->parseJsonFile($composer_lock_path);

    if (!$composer_json || !$composer_lock) {
      throw new \Exception('Unable to parse Composer files.');
    }

    // Get all installed packages with their current information
    $installed_packages = $this->getInstalledPackages($composer_lock);

    // Get latest version information for all packages
    $packages_with_latest = $this->enrichPackagesWithLatestVersions($project_root, $installed_packages);

    return [
      'project_root' => basename($project_root),
      'total_packages' => count($packages_with_latest),
      'packages' => $packages_with_latest,
      'scan_time' => date('Y-m-d H:i:s'),
    ];
  }

  /**
   * Gets all installed packages from composer.lock.
   *
   * @param array $composer_lock
   *   The parsed composer.lock data.
   *
   * @return array
   *   Array of installed packages with their information.
   */
  protected function getInstalledPackages(array $composer_lock): array {
    $packages = [];

    // Process main packages
    if (isset($composer_lock['packages'])) {
      foreach ($composer_lock['packages'] as $package) {
        $packages[$package['name']] = [
          'name' => $package['name'],
          'current_version' => $package['version'],
          'description' => $package['description'] ?? '',
          'type' => $package['type'] ?? 'library',
          'homepage' => $package['homepage'] ?? '',
          'latest_version' => null, // Will be filled later
          'is_outdated' => false, // Will be determined later
        ];
      }
    }

    // Process dev packages if they exist
    if (isset($composer_lock['packages-dev'])) {
      foreach ($composer_lock['packages-dev'] as $package) {
        $packages[$package['name']] = [
          'name' => $package['name'],
          'current_version' => $package['version'],
          'description' => $package['description'] ?? '',
          'type' => $package['type'] ?? 'library',
          'homepage' => $package['homepage'] ?? '',
          'latest_version' => null, // Will be filled later
          'is_outdated' => false, // Will be determined later
          'is_dev' => true,
        ];
      }
    }

    return $packages;
  }

  /**
   * Enriches packages with latest version information using composer show.
   *
   * @param string $project_root
   *   The project root directory.
   * @param array $packages
   *   Array of installed packages.
   *
   * @return array
   *   Array of packages enriched with latest version information.
   */
  protected function enrichPackagesWithLatestVersions(string $project_root, array $packages): array {
    $config = $this->configFactory->get('drupitor_client.settings');
    $timeout = $config->get('command_timeout') ?: 60;
    $logger = $this->loggerFactory->get('drupitor_client');

    // Use composer show command to get latest versions
    $composer_path = $this->getComposerPath();

    // Set environment variables for Composer - FIXED MEMORY LIMIT ISSUE
    $env_vars = [
      'COMPOSER_NO_INTERACTION' => '1',
      'COMPOSER_DISABLE_XDEBUG_WARN' => '1',
      'COMPOSER_CACHE_DIR' => sys_get_temp_dir() . '/composer_cache',
      'PATH' => getenv('PATH'),
      'HOME' => getenv('HOME') ?: '/tmp',
    ];

    // Only set memory limit if we can determine current limit safely
    $current_memory_limit = ini_get('memory_limit');
    if ($current_memory_limit !== false && $current_memory_limit !== '-1') {
      // Convert current limit to bytes and increase it
      $current_bytes = $this->convertToBytes($current_memory_limit);
      $new_limit = max($current_bytes * 2, 512 * 1024 * 1024); // At least 512MB
      $env_vars['COMPOSER_MEMORY_LIMIT'] = $new_limit;
    }

    $command = sprintf(
      'cd %s && %s show --latest --format=json --no-interaction 2>&1',
      escapeshellarg($project_root),
      escapeshellarg($composer_path)
    );

    $logger->info('Executing composer command: @command', ['@command' => $command]);

    // Create descriptor specification
    $descriptorspec = [
      0 => ['pipe', 'r'],  // stdin
      1 => ['pipe', 'w'],  // stdout
      2 => ['pipe', 'w'],  // stderr
    ];

    // Execute command with proper environment
    $process = proc_open($command, $descriptorspec, $pipes, $project_root, $env_vars);

    if (!is_resource($process)) {
      $logger->error('Failed to start composer process');
      return array_values($packages);
    }

    // Close stdin immediately as we don't need to write to it
    fclose($pipes[0]);

    // Make stdout and stderr non-blocking
    stream_set_blocking($pipes[1], false);
    stream_set_blocking($pipes[2], false);

    $start_time = time();
    $output = '';
    $error_output = '';
    $max_execution_time = min($timeout, 120); // Cap at 2 minutes

    // Read from process until it's done or timeout
    while (time() - $start_time < $max_execution_time) {
      $status = proc_get_status($process);

      if (!$status['running']) {
        // Process has finished, read remaining output
        $output .= stream_get_contents($pipes[1]);
        $error_output .= stream_get_contents($pipes[2]);
        break;
      }

      // Read available data without blocking
      $stdout_data = stream_get_contents($pipes[1]);
      $stderr_data = stream_get_contents($pipes[2]);

      if ($stdout_data !== false) {
        $output .= $stdout_data;
      }
      if ($stderr_data !== false) {
        $error_output .= $stderr_data;
      }

      // Small delay to prevent excessive CPU usage
      usleep(250000); // 0.25 seconds
    }

    // Clean up
    fclose($pipes[1]);
    fclose($pipes[2]);
    $return_code = proc_close($process);

    $logger->info('Composer command completed with return code: @code', ['@code' => $return_code]);

    // Filter out memory limit warnings from stderr but log other errors
    $filtered_errors = $this->filterComposerErrors($error_output);
    if (!empty($filtered_errors)) {
      $logger->warning('Composer stderr output: @error', ['@error' => trim($filtered_errors)]);
    }

    // Try to extract JSON even if there were warnings
    $json_output = $this->extractJsonFromOutput($output);

    if ($return_code === 0 && !empty($json_output)) {
      $decoded = json_decode($json_output, TRUE);
      if ($decoded && isset($decoded['installed'])) {
        $logger->info('Successfully parsed composer output with @count packages', ['@count' => count($decoded['installed'])]);

        foreach ($decoded['installed'] as $package_info) {
          $package_name = $package_info['name'];
          if (isset($packages[$package_name])) {
            $packages[$package_name]['latest_version'] = $package_info['latest'] ?? $package_info['version'];
            // Determine if package is outdated
            if (isset($package_info['latest']) && $package_info['latest'] !== $package_info['version']) {
              $packages[$package_name]['is_outdated'] = true;
            }
          }
        }
      } else {
        $logger->warning('Failed to parse composer JSON output or no installed packages found');
      }
    } else {
      $logger->warning('Composer command failed or produced no output. Return code: @code', ['@code' => $return_code]);

      // Log first part of output for debugging
      if (!empty($output)) {
        $logger->info('Composer stdout (first 500 chars): @output', ['@output' => substr($output, 0, 500)]);
      }
    }

    // Convert to indexed array and sort by name
    $result = array_values($packages);
    usort($result, function($a, $b) {
      return strcmp($a['name'], $b['name']);
    });

    return $result;
  }

  /**
   * Converts memory limit string to bytes.
   *
   * @param string $limit
   *   Memory limit string (e.g., '256M', '1G').
   *
   * @return int
   *   Memory limit in bytes.
   */
  protected function convertToBytes(string $limit): int {
    $limit = trim($limit);
    $last = strtolower($limit[strlen($limit) - 1]);
    $value = (int) $limit;

    switch ($last) {
      case 'g':
        $value *= 1024;
      case 'm':
        $value *= 1024;
      case 'k':
        $value *= 1024;
    }

    return $value;
  }

  /**
   * Filters out known harmless errors from Composer output.
   *
   * @param string $error_output
   *   The stderr output from Composer.
   *
   * @return string
   *   Filtered error output.
   */
  protected function filterComposerErrors(string $error_output): string {
    if (empty($error_output)) {
      return '';
    }

    $lines = explode("\n", $error_output);
    $filtered_lines = [];

    foreach ($lines as $line) {
      $line = trim($line);
      if (empty($line)) {
        continue;
      }

      // Skip memory limit warnings
      if (strpos($line, 'Failed to set memory limit') !== false) {
        continue;
      }
      if (strpos($line, 'memory_limit') !== false && strpos($line, 'Warning') !== false) {
        continue;
      }

      // Skip other common harmless warnings
      if (strpos($line, 'Xdebug') !== false) {
        continue;
      }

      $filtered_lines[] = $line;
    }

    return implode("\n", $filtered_lines);
  }

  /**
   * Extracts JSON from Composer output, ignoring warnings.
   *
   * @param string $output
   *   The full output from Composer.
   *
   * @return string
   *   Extracted JSON string.
   */
  protected function extractJsonFromOutput(string $output): string {
    if (empty($output)) {
      return '';
    }

    // Look for JSON starting with { and ending with }
    $json_start = strpos($output, '{');
    if ($json_start === false) {
      return '';
    }

    // Find the last closing brace
    $json_end = strrpos($output, '}');
    if ($json_end === false || $json_end <= $json_start) {
      return '';
    }

    return substr($output, $json_start, $json_end - $json_start + 1);
  }

  /**
   * Gets the Composer executable path.
   *
   * @return string
   *   Path to composer executable.
   */
  protected function getComposerPath(): string {
    $config = $this->configFactory->get('drupitor_client.settings');
    $composer_path = $config->get('composer_path') ?: 'composer';

    // Validate composer path for security
    if (!preg_match('/^[a-zA-Z0-9\/_.-]+$/', $composer_path)) {
      return 'composer'; // Fallback to system composer
    }

    return $composer_path;
  }

  /**
   * Gets the project root directory.
   *
   * @return string|false
   *   Project root path or FALSE on failure.
   */
  protected function getProjectRoot(): string|false {
    $drupal_root = DRUPAL_ROOT;

    // Look for composer.json in parent directories
    $current_dir = $drupal_root;
    for ($i = 0; $i < 5; $i++) {
      if (file_exists($current_dir . '/composer.json')) {
        return realpath($current_dir);
      }
      $parent_dir = dirname($current_dir);
      if ($parent_dir === $current_dir) {
        break;
      }
      $current_dir = $parent_dir;
    }

    return FALSE;
  }

  /**
   * Validates if a path is safe to use.
   *
   * @param string $path
   *   The path to validate.
   *
   * @return bool
   *   TRUE if path is safe, FALSE otherwise.
   */
  protected function isValidPath(string $path): bool {
    // Check for directory traversal attempts
    if (strpos($path, '..') !== FALSE) {
      return FALSE;
    }

    // Check for null bytes
    if (strpos($path, "\0") !== FALSE) {
      return FALSE;
    }

    // Ensure path is within expected boundaries
    $realpath = realpath(dirname($path));
    if (!$realpath) {
      return FALSE;
    }

    // Check if within project boundaries
    $drupal_root_real = realpath(DRUPAL_ROOT);
    $project_root_real = realpath($this->getProjectRoot());

    return ($realpath === $project_root_real ||
      strpos($realpath, $drupal_root_real) === 0 ||
      strpos($realpath, $project_root_real) === 0);
  }

  /**
   * Safely parses a JSON file.
   *
   * @param string $file_path
   *   Path to the JSON file.
   *
   * @return array|false
   *   Parsed JSON data or FALSE on failure.
   */
  protected function parseJsonFile(string $file_path): array|false {
    if (!is_readable($file_path)) {
      return FALSE;
    }

    $content = file_get_contents($file_path);
    if ($content === FALSE) {
      return FALSE;
    }

    // Limit file size for security (10MB max)
    if (strlen($content) > 10 * 1024 * 1024) {
      return FALSE;
    }

    $decoded = json_decode($content, TRUE);
    if (json_last_error() !== JSON_ERROR_NONE) {
      return FALSE;
    }

    return $decoded;
  }

  /**
   * Encrypts data using AES-256-CBC.
   *
   * @param array $data
   *   The data to encrypt.
   *
   * @return array
   *   Array containing encrypted data and IV.
   *
   * @throws \Exception
   */
  protected function encryptData(array $data): array {
    $config = $this->configFactory->get('drupitor_client.settings');
    $encryption_key = $config->get('encryption_key');
    $method = $config->get('encryption_method') ?: 'AES-256-CBC';

    if (empty($encryption_key)) {
      throw new \Exception('Encryption key not configured.');
    }

    // Ensure key is the right length for AES-256
    $key = hash('sha256', $encryption_key, true);

    // Convert data to JSON
    $json_data = json_encode($data);
    if ($json_data === false) {
      throw new \Exception('Failed to encode data as JSON.');
    }

    // Generate random IV
    $iv_length = openssl_cipher_iv_length($method);
    $iv = openssl_random_pseudo_bytes($iv_length);

    // Encrypt the data - handle GCM mode differently
    if ($method === 'AES-256-GCM') {
      $tag = '';
      $encrypted = openssl_encrypt($json_data, $method, $key, 0, $iv, $tag);
      if ($encrypted === false) {
        throw new \Exception('Encryption failed.');
      }

      return [
        'payload' => base64_encode($encrypted),
        'iv' => base64_encode($iv),
        'tag' => base64_encode($tag),
        'method' => $method,
      ];
    } else {
      $encrypted = openssl_encrypt($json_data, $method, $key, 0, $iv);
      if ($encrypted === false) {
        throw new \Exception('Encryption failed.');
      }

      return [
        'payload' => base64_encode($encrypted),
        'iv' => base64_encode($iv),
        'method' => $method,
      ];
    }
  }

  /**
   * Decrypts data (for testing purposes).
   *
   * @param array $encrypted_data
   *   The encrypted data array.
   *
   * @return array
   *   The decrypted data.
   *
   * @throws \Exception
   */
  protected function decryptData(array $encrypted_data): array {
    $config = $this->configFactory->get('drupitor_client.settings');
    $encryption_key = $config->get('encryption_key');

    if (empty($encryption_key)) {
      throw new \Exception('Encryption key not configured.');
    }

    $key = hash('sha256', $encryption_key, true);
    $method = $encrypted_data['method'];
    $encrypted = base64_decode($encrypted_data['payload']);
    $iv = base64_decode($encrypted_data['iv']);

    // Handle GCM mode differently
    if ($method === 'AES-256-GCM') {
      if (!isset($encrypted_data['tag'])) {
        throw new \Exception('Authentication tag missing for GCM decryption.');
      }
      $tag = base64_decode($encrypted_data['tag']);
      $decrypted = openssl_decrypt($encrypted, $method, $key, 0, $iv, $tag);
    } else {
      $decrypted = openssl_decrypt($encrypted, $method, $key, 0, $iv);
    }

    if ($decrypted === false) {
      throw new \Exception('Decryption failed.');
    }

    return json_decode($decrypted, true);
  }

}