<?php

namespace Drupal\drupitor_client\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Configuration form for Drupitor Client module.
 */
class DrupitorClientConfigForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['drupitor_client.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'drupitor_client_config_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $config = $this->config('drupitor_client.settings');

    $form['enabled'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable Drupitor client'),
      '#description' => $this->t('Enable the Drupitor client API endpoint. Disable this to completely block access to update information. The field is not required, but the Drupitor client will not function without it.'),
      '#default_value' => $config->get('enabled') ?? FALSE,
    ];

    // API Configuration
    $form['api'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('API configuration'),
    ];

    $form['api']['api_token'] = [
      '#type' => 'textfield',
      '#title' => $this->t('API token'),
      '#description' => $this->t('The API token is provided by the Drupitor host on instance creation. The token is used to authenticate requests to the Drupitor API endpoint. Keep this token secure and do not share it publicly. The field is not required, but the Drupitor client will not function without it.'),
      '#default_value' => $config->get('api_token'),
      '#attributes' => ['autocomplete' => 'off'],
      '#maxlength' => 255,
    ];

    // Encryption settings
    $form['encryption'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Encryption configuration'),
    ];

    $form['encryption']['encryption_key'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Encryption key'),
      '#description' => $this->t('The encryption key is provided by the Drupitor host on instance creation and is used to encrypt the sensitive data on the API endpoint and to decrypt it on the Drupitor host. This key should be kept secret and secure.'),
      '#default_value' => $config->get('encryption_key'),
      '#attributes' => ['autocomplete' => 'off'],
    ];

    $form['encryption']['encryption_method'] = [
      '#type' => 'select',
      '#title' => $this->t('Encryption method'),
      '#description' => $this->t('Select the encryption method to use for sensitive data. <strong>AES-256-GCM</strong> is recommended for its security and performance.'),
      '#options' => [
        'AES-256-CBC' => 'AES-256-CBC',
        'AES-256-GCM' => 'AES-256-GCM (recommended)',
      ],
      '#default_value' => $config->get('encryption_method') ?: 'AES-256-GCM',
    ];

    $form['composer_path'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Composer path'),
      '#description' => $this->t('Path to the Composer executable. Leave empty to use system default "composer".'),
      '#default_value' => $config->get('composer_path') ?? '',
      '#placeholder' => 'composer',
    ];

    $form['command_timeout'] = [
      '#type' => 'number',
      '#title' => $this->t('Command timeout'),
      '#description' => $this->t('Timeout in seconds for Composer commands. Default is 60 seconds.'),
      '#default_value' => $config->get('command_timeout') ?? 60,
      '#min' => 10,
      '#max' => 300,
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    $composer_path = $form_state->getValue('composer_path');
    $api_token = $form_state->getValue('api_token');

    // Validate composer path if provided
    if (!empty($composer_path)) {
      if (!preg_match('/^[a-zA-Z0-9\/_.-]+$/', $composer_path)) {
        $form_state->setErrorByName('composer_path', $this->t('Composer path contains invalid characters. Only alphanumeric characters, forward slashes, hyphens, periods, and underscores are allowed.'));
      }
    }

    // Validate API token
    if (!empty($api_token)) {
      if (strlen($api_token) < 32) {
        $form_state->setErrorByName('api_token', $this->t('API token must be at least 32 characters long for security.'));
      }
      if (!preg_match('/^[a-zA-Z0-9._-]+$/', $api_token)) {
        $form_state->setErrorByName('api_token', $this->t('API token contains invalid characters. Only alphanumeric characters, dots, hyphens, and underscores are allowed.'));
      }
    }

    parent::validateForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $this->config('drupitor_client.settings')
      ->set('enabled', $form_state->getValue('enabled'))
      ->set('api_token', $form_state->getValue('api_token'))
      ->set('composer_path', $form_state->getValue('composer_path'))
      ->set('command_timeout', $form_state->getValue('command_timeout'))
      ->set('encryption_key', $form_state->getValue('encryption_key'))
      ->set('encryption_method', $form_state->getValue('encryption_method'))
      ->save();

    parent::submitForm($form, $form_state);
  }

}