<?php

namespace Drupal\drush_extra\Drush\Commands\Debug;

use Consolidation\AnnotatedCommand\CommandData;
use Consolidation\AnnotatedCommand\Hooks\HookManager;
use Drupal\Core\Entity\EntityTypeBundleInfoInterface;
use Drupal\Core\Entity\EntityTypeRepositoryInterface;
use Drupal\drush_extra\Services\CommandInfoService;
use Drupal\drush_extra\Services\DrushTableOutputService;
use Drush\Attributes as CLI;
use Drush\Commands\AutowireTrait;
use Drush\Commands\DrushCommands;
use Symfony\Component\DependencyInjection\Attribute\Autowire;

/**
 * Drush command to display all entities and bundles.
 */
class EntityCommand extends DrushCommands {

  use AutowireTrait;

  public const COMMAND = 'debug:entity';
  public const ALIASES = ['debe'];

  public function __construct(
    private EntityTypeRepositoryInterface $entityTypeRepository,
    private EntityTypeBundleInfoInterface $entityTypeBundle,
    #[Autowire(service: 'drush_extra.drush_table')]
    private DrushTableOutputService $tableHelper,
  ) {
    parent::__construct();
  }

  /**
   * Displays all entities and bundles.
   *
   * @param string $entity_group
   *   Entity group, e.g. Content (optional)
   */
  #[CLI\Command(name: self::COMMAND, aliases: self::ALIASES)]
  #[CLI\Argument(name: 'entity_group', description: 'Entity group, e.g. Content (optional)')]
  #[CLI\Usage('debug:entity', 'Displays all entities and bundles')]
  #[CLI\Usage('debug:entity Content', 'Displays all entities and bundles for the Content group')]
  public function entity(?string $entity_group = NULL) {
    $this->tableHelper->addHeaderRow([
      dt('Entity class ID'),
      dt('Entity ID'),
      dt('Entity label'),
      dt('Bundle'),
      dt('Entity group'),
    ]);

    $entityTypes = $this->entityTypeRepository->getEntityTypeLabels(TRUE);

    if ($entity_group && !in_array($entity_group, array_keys($entityTypes))) {
      return;
    }

    if ($entity_group) {
      $entityGroups = [$entity_group];
    }
    else {
      $entityGroups = array_keys($entityTypes);
    }

    foreach ($entityGroups as $entityGroup) {
      $entities = $entityTypes[$entityGroup];
      ksort($entities);

      foreach ($entities as $entityId => $entityType) {
        $this->tableHelper->addRow([
          $entityId,
          $entityId,
          $entityType->render(),
          '',
          $entityGroup,
        ], $entityId);

        $entityBundles = $this->entityTypeBundle->getBundleInfo($entityId);

        foreach ($entityBundles as $bundleId => $bundle) {
          $this->tableHelper->addRow([
            $entityId,
            $bundleId,
            $bundle['label'],
            'yes',
            $entityGroup,
          ], $bundleId);
        }
      }
    }

    $this->io()->table(
      $this->tableHelper->getHeaderRows(),
      $this->tableHelper->getRows()
    );
  }

  /**
   * Displays the command description before execution.
   */
  #[CLI\Hook(type: HookManager::PRE_COMMAND_HOOK, target: self::COMMAND)]
  public function preCommand(CommandData $commandData) {
    $commandHelper = new CommandInfoService($commandData);

    $this->io()->text(
      $commandHelper->getCommandDescription()
    );
  }

}
