<?php

namespace Drupal\drush_extra\Drush\Commands\Debug;

use Consolidation\AnnotatedCommand\CommandData;
use Consolidation\AnnotatedCommand\Hooks\HookManager;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\drush_extra\Services\CommandInfoService;
use Drupal\drush_extra\Services\DrushTableOutputService;
use Drupal\image\ImageEffectPluginCollection;
use Drush\Attributes as CLI;
use Drush\Commands\AutowireTrait;
use Drush\Commands\DrushCommands;
use Symfony\Component\DependencyInjection\Attribute\Autowire;

/**
 * Drush command to display all image styles and their effects.
 */
class ImageStylesCommand extends DrushCommands {

  use AutowireTrait;

  public const COMMAND = 'debug:image:styles';
  public const ALIASES = ['debis'];

  public function __construct(
    private EntityTypeManagerInterface $entityTypeManager,
    #[Autowire(service: 'drush_extra.drush_table')]
    private readonly DrushTableOutputService $tableHelper,
  ) {
    parent::__construct();
  }

  /**
   * Displays all images styles with effects.
   */
  #[CLI\Command(name: self::COMMAND, aliases: self::ALIASES)]
  #[CLI\Usage('debug:image:styles', 'Displays all images styles with effects')]
  public function styles() {
    $imageStyles = $this->entityTypeManager->getStorage('image_style')->loadMultiple();
    $this->imageStylesList($imageStyles);
  }

  /**
   * Displays a list of image styles with their effects.
   *
   * @param array $imageStyles
   *   An array of image styles, keyed by their machine names.
   *   Each value is an instance of \Drupal\image\ImageStyleInterface.
   */
  protected function imageStylesList(array $imageStyles) {
    $this->tableHelper->addHeaderRow([
      dt('Machine Name'),
      dt('Label'),
      dt('Effects'),
    ]);

    /** @var \Drupal\image\ImageStyleInterface $style */
    foreach ($imageStyles as $styleId => $style) {
      $this->tableHelper->addRow([
        $styleId,
        $style->label(),
        '',
      ]);

      $this->styleEffectsList($style->getEffects());
    }

    $this->io()->table(
      $this->tableHelper->getHeaderRows(),
      $this->tableHelper->getRows()
    );
  }

  /**
   * Displays a list of image style effects.
   *
   * @param \Drupal\image\ImageEffectPluginCollection $styleEffects
   *   An instance of ImageEffectPluginCollection containing the effects.
   */
  protected function styleEffectsList(ImageEffectPluginCollection $styleEffects) {
    /** @var \Drupal\image\ImageEffectInterface $effect */
    foreach ($styleEffects as $effect) {
      $effectSummary = $effect->getSummary();

      $this->tableHelper->addRowWithOnlyLastColumn(
        sprintf(
          "%s / %s",
          $effectSummary['#effect']['id'],
          $effect->label(),
        )
      );

      if (array_key_exists('#markup', $effectSummary)) {
        $markup = $effectSummary['#markup'];

        if (!empty($markup)) {
          $this->tableHelper->addRowWithOnlyLastColumn($markup);
        }
      }

      if (array_key_exists('#data', $effectSummary)) {
        $this->tableHelper->addRowWithOnlyLastColumn(
          json_encode($effectSummary['#data'])
        );
      }

      $this->tableHelper->addEmptyRow();
    }
  }

  /**
   * Displays the command description before executing the command.
   */
  #[CLI\Hook(type: HookManager::PRE_COMMAND_HOOK, target: self::COMMAND)]
  public function preCommand(CommandData $commandData) {
    $commandHelper = new CommandInfoService($commandData);

    $this->io()->text(
      $commandHelper->getCommandDescription()
    );
  }

}
