<?php

namespace Drupal\drush_extra\Drush\Commands\Node;

use Consolidation\AnnotatedCommand\CommandData;
use Consolidation\AnnotatedCommand\Hooks\HookManager;
use Drupal\Core\State\StateInterface;
use Drupal\drush_extra\Services\CommandInfoService;
use Drush\Attributes as CLI;
use Drush\Commands\AutowireTrait;
use Drush\Commands\DrushCommands;
use function node_access_rebuild;

/**
 * Drush command to rebuild node access permissions.
 */
class AccessRebuildCommand extends DrushCommands {

  use AutowireTrait;

  public function __construct(
    private StateInterface $state,
  ) {
    parent::__construct();
  }

  const COMMAND = 'node:access:rebuild';
  const ALIASES = ['nar'];

  /**
   * Rebuild node access permissions.
   */
  #[CLI\Command(name: self::COMMAND, aliases: self::ALIASES)]
  #[CLI\Usage(name: 'node:access:rebuild', description: 'Rebuild node access permissions')]
  #[CLI\Usage(name: 'node:access:rebuild batch', description: 'Process in batch mode')]
  public function rebuild($batch = NULL) {
    try {
      node_access_rebuild((bool) $batch);
    }
    catch (\Exception $e) {
      $this->io()->error($e->getMessage());
      return;
    }

    $needs_rebuild = $this->state->get('node.node_access_needs_rebuild') ?: FALSE;
    if ($needs_rebuild) {
      $this->io()->error(
        dt('Rebuilding permissions was not successful')
      );
      return;
    }

    $this->io()->success(
      dt('Done rebuilding permissions')
    );
  }

  /**
   * Displays the command description before executing the command.
   */
  #[CLI\Hook(type: HookManager::PRE_COMMAND_HOOK, target: self::COMMAND)]
  public function preCommand(CommandData $commandData) {
    $commandHelper = new CommandInfoService($commandData);

    $this->io()->text(
      $commandHelper->getCommandDescription()
    );
  }

}
