<?php

namespace Drupal\drush_extra\Services;

/**
 * Helper class for managing table data in Drush commands.
 */
class DrushTableOutputService {
  /**
   * The header rows of the table.
   *
   * @var array
   */
  protected $headerRows;

  /**
   * The data rows of the table.
   *
   * @var array
   */
  protected $rows;

  /**
   * The number of columns in the table.
   *
   * @var int
   */
  protected $columnsCount;

  public function __construct() {
    $this->columnsCount = 0;
    $this->headerRows = [];
    $this->rows = [];
  }

  /**
   * Returns the number of columns in the table.
   */
  public function getHeaderRows(): array {
    return $this->headerRows;
  }

  /**
   * Returns the number of columns in the table.
   */
  public function getRows(): array {
    return $this->rows;
  }

  /**
   * Returns the number of columns in the table.
   */
  public function addHeaderRow(array $row) {
    $this->headerRows[] = $row;

    $this->calculateColumnsCount();
  }

  /**
   * Adds a column to the first header row.
   */
  public function addHeaderRowColumn(string $column) {
    $this->headerRows[0][] = $column;
  }

  /**
   * Adds a row to the table.
   */
  public function addRow(array $row, string|int|null $key = NULL) {
    if ($key) {
      $this->rows[$key] = $row;
    }
    else {
      $this->rows[] = $row;
    }

    $this->calculateColumnsCount();
  }

  /**
   * Adds a column to a specific row.
   */
  public function addRowColumn(string $column, string|int $key) {
    $this->rows[$key][] = $column;
  }

  /**
   * Adds an empty row to the table.
   */
  public function addEmptyRow() {
    $row = array_fill(0, $this->getRowSize(), '');

    $this->addRow($row);
  }

  /**
   * Adds a row with only the last column filled.
   */
  public function addRowWithOnlyLastColumn(string|int $columnContent) {
    $row = array_fill(0, $this->getRowSize(), '');
    $row[$this->getRowSize()] = $columnContent;

    $this->addRow($row);
  }

  /**
   * Calculates the number of columns in the table.
   */
  private function calculateColumnsCount() {
    if ($this->columnsCount !== 0) {
      return;
    }

    if (!empty($this->headerRows)) {
      $this->columnsCount = count($this->headerRows[0]);
      return;
    }

    $this->columnsCount = max(array_map('count', $this->rows));
  }

  /**
   * Returns the number of columns in the table.
   */
  private function getRowSize() {
    return $this->columnsCount > 0 ? $this->columnsCount - 1 : 0;
  }

}
