<?php

declare(strict_types=1);

namespace Drupal\drush_queue_run_all\Drush\Commands;

use Consolidation\AnnotatedCommand\Hooks\HookManager;
use Drupal\Component\Datetime\TimeInterface;
use Drupal\Core\KeyValueStore\KeyValueFactoryInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Queue\DelayableQueueInterface;
use Drupal\Core\Queue\DelayedRequeueException;
use Drupal\Core\Queue\QueueFactory;
use Drupal\Core\Queue\QueueGarbageCollectionInterface;
use Drupal\Core\Queue\QueueInterface;
use Drupal\Core\Queue\QueueWorkerInterface;
use Drupal\Core\Queue\QueueWorkerManagerInterface;
use Drupal\Core\Queue\RequeueException;
use Drupal\Core\Queue\SuspendQueueException;
use Drush\Attributes as CLI;
use Drush\Commands\AutowireTrait;
use Drush\Commands\DrushCommands;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\DependencyInjection\Attribute\Autowire;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Drush command for running all available queues.
 */
class QueueRunAllCommands extends DrushCommands {

  use AutowireTrait;

  const RUN_ALL = 'queue:run-all';

  /**
   * Default configuration for queue processing.
   */
  protected array $queueConfig = [
    'suspendMaximumWait' => 30.0,
  ];

  /**
   * Static cache of queue definitions.
   */
  protected static array $queues;

  public function __construct(
    protected LoggerChannelFactoryInterface $loggerFactory,
    protected QueueWorkerManagerInterface $workerManager,
    #[Autowire(service: 'keyvalue')]
    protected KeyValueFactoryInterface $keyValueFactory,
    protected ContainerInterface $container,
    protected QueueFactory $queueService,
    protected TimeInterface $time,
  ) {
    parent::__construct();

    // Drush does not yet support autowiring of parameters.
    // @see https://github.com/drush-ops/drush/issues/6357
    if ($this->container->hasParameter('queue.config')) {
      $this->queueConfig = $this->container->getParameter('queue.config') + $this->queueConfig;
    }
  }

  /**
   * Run all available queues.
   */
  #[CLI\Command(name: self::RUN_ALL, aliases: ['queue-run-all'])]
  #[CLI\Option(name: 'time-limit', description: 'The maximum number of seconds allowed to run the queue.')]
  #[CLI\Option(name: 'items-limit', description: 'The maximum number of items allowed to run the queue.')]
  #[CLI\Option(name: 'memory-limit', description: 'The maximum amount of memory the script can consume before exiting. Can be a value supported by the memory_limit PHP setting or a percentage.')]
  #[CLI\Option(name: 'lease-time', description: 'The maximum number of seconds that an item remains claimed.')]
  #[CLI\Option(name: 'daemon', description: 'Keep the command running indefinitely, or until one of the chosen limits has been reached.')]
  #[CLI\Option(name: 'queues', description: 'Comma-separated list of queue machine names to process.')]
  #[CLI\Option(name: 'exclude-queues', description: 'Comma-separated list of queue machine names to exclude from processing.')]
  public function runAll(
    $options = [
      'time-limit' => self::REQ,
      'items-limit' => self::REQ,
      'memory-limit' => self::REQ,
      'lease-time' => self::REQ,
      'daemon' => FALSE,
      'queues' => self::REQ,
      'exclude-queues' => self::REQ,
    ],
  ): void {
    $start = microtime(TRUE);
    $end = time() + $options['time-limit'];
    $time_remaining = $options['time-limit'];
    $items_count = 0;

    // Filter queues according to the provided options.
    $queues = $this->filterQueues(
      $this->getQueues(),
      $options['queues'],
      $options['exclude-queues']
    );

    $delays = $this->keyValueFactory->get('queue_run_all_delays');
    $maxDelay = (float) $this->queueConfig['suspendMaximumWait'];

    do {
      foreach ($queues as $name => $info) {
        if ($delays->has($name)) {
          if ($delays->get($name) > $this->time->getCurrentMicroTime()) {
            // Still delaying this queue.
            continue;
          }
          else {
            // Delay has passed; remove it.
            $delays->delete($name);
          }
        }

        $queue = $this->queueService->get($name);
        $worker = $this->workerManager->createInstance($name);
        $lease_time = $options['lease-time'] ?? $info['cron']['time'] ?? 30;
        $queue_starting = TRUE;
        $queue_start = microtime(TRUE);
        $queue_items_count = 0;

        if ($queue instanceof QueueGarbageCollectionInterface) {
          $queue->garbageCollection();
        }

        while (!$this->hasReachedLimit($options, $items_count, $time_remaining) && $item = $queue->claimItem($lease_time)) {
          if ($queue_starting) {
            $this->logger()->notice('Processing queue ' . $name);
          }

          try {
            if ($this->processItem($queue, $worker, $name, $item)) {
              $queue_items_count++;
            }
          }
          catch (SuspendQueueException $e) {
            // Skip to the next queue, delaying next processing if requested.
            if ($e->isDelayable()) {
              $delay = min($e->getDelay(), $maxDelay);
              $delays->set($name, $this->time->getCurrentMicroTime() + $delay);
            }
            break;
          }

          $time_remaining = $end - time();
          $queue_starting = FALSE;
        }

        if ($queue_items_count > 0) {
          $items_count += $queue_items_count;
          $elapsed = microtime(TRUE) - $queue_start;
          $this->logger()->success(dt('Processed @count items from the @name queue in @elapsed sec.', [
            '@count' => $queue_items_count,
            '@name' => $name,
            '@elapsed' => round($elapsed, 2),
          ]));
        }
      }
      if ($options['daemon']) {
        sleep(1);
      }
    } while ($options['daemon'] && !$this->hasReachedLimit($options, $items_count, $time_remaining));

    $elapsed = microtime(TRUE) - $start;
    $this->logger()->success(dt('Processed @count items in @elapsed sec.', [
      '@count' => $items_count,
      '@elapsed' => round($elapsed, 2),
    ]));
  }

  /**
   * Post-initialize hook for the run-all command.
   */
  #[CLI\Hook(type: HookManager::POST_INITIALIZE, target: self::RUN_ALL)]
  public function postInitRunAll(InputInterface $input): void {
    $input->setOption('time-limit', (int) $input->getOption('time-limit'));
    $input->setOption('items-limit', (int) $input->getOption('items-limit'));
    $input->setOption('memory-limit', $this->parseMemoryLimit((string) $input->getOption('memory-limit')));

    // Validate that --queues and --exclude-queues are not used together.
    $queues = $input->getOption('queues');
    $excludeQueues = $input->getOption('exclude-queues');
    if ($queues !== NULL && $excludeQueues !== NULL) {
      throw new \InvalidArgumentException('The --queues and --exclude-queues options cannot be used together.');
    }
  }

  /**
   * Filter queues based on --queues or --exclude-queues options.
   *
   * @param array $queues
   *   All available queue definitions.
   * @param string|null $include
   *   Comma-separated list of queue names to include.
   * @param string|null $exclude
   *   Comma-separated list of queue names to exclude.
   *
   * @return array
   *   Filtered queue definitions.
   */
  protected function filterQueues(array $queues, ?string $include, ?string $exclude): array {
    if ($include !== NULL) {
      $allowed = array_map('trim', explode(',', $include));
      return array_intersect_key($queues, array_flip($allowed));
    }

    if ($exclude !== NULL) {
      $forbidden = array_map('trim', explode(',', $exclude));
      return array_diff_key($queues, array_flip($forbidden));
    }

    return $queues;
  }

  /**
   * Check if the queue processing has reached the limit.
   *
   * @param array $options
   *   The command options.
   * @param int $items_count
   *   The number of items processed.
   * @param int $time_remaining
   *   The remaining time.
   */
  protected function hasReachedLimit(array $options, int $items_count, int $time_remaining): bool {
    return ($options['time-limit'] && $time_remaining <= 0)
      || ($options['items-limit'] && $items_count >= $options['items-limit'])
      || ($options['memory-limit'] && memory_get_usage() >= $options['memory-limit']);
  }

  /**
   * Parse the memory-limit option value.
   */
  protected function parseMemoryLimit(?string $value): ?int {
    if ($value === NULL || $value === '') {
      return NULL;
    }

    $last = strtolower($value[strlen($value) - 1]);
    $size = (int) rtrim($value, 'GgMmKk%');

    switch ($last) {
      case 'g':
        $size *= DRUSH_KILOBYTE;
      case 'm':
        $size *= DRUSH_KILOBYTE;
      case 'k':
        $size *= DRUSH_KILOBYTE;
    }

    if ($last === '%') {
      $size = (int) ($size * (drush_memory_limit() / 100));
    }

    return $size;
  }

  /**
   * Process an item from the queue.
   *
   * @return bool
   *   TRUE if the item was processed, FALSE otherwise.
   */
  protected function processItem(QueueInterface $queue, QueueWorkerInterface $worker, string $name, object $item): bool {
    try {
      // @phpstan-ignore-next-line
      $this->logger()->info(dt('Processing item @id from @name queue.', [
        '@name' => $name,
        '@id' => $item->item_id ?? $item->qid,
      ]));
      // @phpstan-ignore-next-line
      $worker->processItem($item->data);
      $queue->deleteItem($item);
      return TRUE;
    }
    catch (RequeueException) {
      // The worker requested the task to be immediately requeued.
      $queue->releaseItem($item);
    }
    catch (SuspendQueueException $e) {
      // If the worker indicates the whole queue should be skipped, release
      // the item and go to the next queue.
      $queue->releaseItem($item);

      $maxDelay = (float) $this->queueConfig['suspendMaximumWait'];
      if ($e->isDelayable()) {
        $delay = min($e->getDelay(), $maxDelay);
        $this->logger()->notice(dt('The queue worker suspended further processing of the queue for %delay seconds.', ['%delay' => $delay]));
      }
      else {
        $this->logger()->notice(dt('The queue worker suspended further processing of the queue.'));
      }

      // Skip to the next queue.
      throw $e;
    }
    catch (DelayedRequeueException $e) {
      // The worker requested the task not be immediately re-queued.
      // - If the queue doesn't support ::delayItem(), we should leave the
      // item's current expiry time alone.
      // - If the queue does support ::delayItem(), we should allow the
      // queue to update the item's expiry using the requested delay.
      if ($queue instanceof DelayableQueueInterface) {
        // This queue can handle a custom delay; use the duration provided
        // by the exception.
        $queue->delayItem($item, $e->getDelay());
      }
    }
    catch (\Exception $e) {
      // In case of any other kind of exception, log it and leave the
      // item in the queue to be processed again later.
      $this->logger()->error($e->getMessage());
      $this->loggerFactory->get('drush_queue_run_all')->error($e->getMessage());
    }

    return FALSE;
  }

  /**
   * Get the queue definitions.
   */
  public function getQueues(): array {
    if (!isset(static::$queues)) {
      static::$queues = [];
      foreach ($this->workerManager->getDefinitions() as $name => $info) {
        static::$queues[$name] = $info;
      }
    }
    return static::$queues;
  }

}
