<?php

namespace Drupal\drw\Plugin\Field\FieldWidget;

use Drupal\Core\Datetime\DrupalDateTime;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\datetime\Plugin\Field\FieldWidget\DateTimeWidgetBase;

/**
 * Plugin implementation of the 'drw_date_range' widget.
 *
 * @FieldWidget(
 *   id = "drw_date_range",
 *   label = @Translation("Date with range validation"),
 *   field_types = {
 *     "datetime"
 *   }
 * )
 */
class DateRangeWidget extends DateTimeWidgetBase
{

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings()
  {
    return [
      'min_date' => '',
      'max_date' => '',
      'min_error_message' => 'The date must be on or after @min.',
      'max_error_message' => 'The date must be on or before @max.',
    ] + parent::defaultSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state)
  {
    $element = parent::settingsForm($form, $form_state);

    $element['min_date'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Minimum date'),
      '#default_value' => $this->getSetting('min_date'),
      '#description' => $this->t('Enter a date or relative format. Examples: "today", "-18 years", "2000-01-01", "-2 weeks". Leave empty for no minimum.'),
    ];

    $element['max_date'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Maximum date'),
      '#default_value' => $this->getSetting('max_date'),
      '#description' => $this->t('Enter a date or relative format. Examples: "today", "+1 year", "2025-12-31", "+3 months". Leave empty for no maximum.'),
    ];

    $element['min_error_message'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Minimum date error message'),
      '#default_value' => $this->getSetting('min_error_message'),
      '#description' => $this->t('Error message when date is before minimum. Use @min as placeholder for the minimum date.'),
    ];

    $element['max_error_message'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Maximum date error message'),
      '#default_value' => $this->getSetting('max_error_message'),
      '#description' => $this->t('Error message when date is after maximum. Use @max as placeholder for the maximum date.'),
    ];

    return $element;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary()
  {
    $summary = parent::settingsSummary();

    $min_date = $this->getSetting('min_date');
    $max_date = $this->getSetting('max_date');

    if (!empty($min_date)) {
      $summary[] = $this->t('Minimum date: @min', ['@min' => $min_date]);
    }

    if (!empty($max_date)) {
      $summary[] = $this->t('Maximum date: @max', ['@max' => $max_date]);
    }

    if (empty($min_date) && empty($max_date)) {
      $summary[] = $this->t('No date range restrictions');
    }

    return $summary;
  }

  /**
   * {@inheritdoc}
   */
  public function formElement(FieldItemListInterface $items, $delta, array $element, array &$form, FormStateInterface $form_state)
  {
    $element = parent::formElement($items, $delta, $element, $form, $form_state);

    // Pass the title to the value sub-element so it reaches datetime_wrapper template.
    if (!empty($element['#title']) && isset($element['value'])) {
      $element['value']['#title'] = $element['#title'];
    }

    // Force date-only format (no time or seconds).
    $element['value']['#date_time_element'] = 'none';
    $element['value']['#date_time_format'] = '';

    // Add data attribute with field name for identification.
    $field_name = $items->getName();
    $element['#attributes']['data-date-range-field'] = $field_name;
    $element['value']['#attributes']['data-date-range-field'] = $field_name;

    // Add HTML5 min and max attributes.
    $min_date = $this->getSetting('min_date');
    $max_date = $this->getSetting('max_date');

    if (!empty($min_date)) {
      $min_datetime = $this->parseRelativeDate($min_date);

      if ($min_datetime) {
        $element['value']['#attributes']['min'] = $min_datetime->format('Y-m-d');
      }
    }

    if (!empty($max_date)) {
      $max_datetime = $this->parseRelativeDate($max_date);
      if ($max_datetime) {
        $element['value']['#attributes']['max'] = $max_datetime->format('Y-m-d');
      }
    }

    return $element;
  }

  /**
   * Parse relative date formats.
   *
   * @param string $date_string
   *   The date string to parse.
   *
   * @return \Drupal\Core\Datetime\DrupalDateTime|null
   *   The parsed date object or NULL on error.
   */
  protected function parseRelativeDate($date_string)
  {
    if (empty($date_string)) {
      return NULL;
    }

    $date = new DrupalDateTime($date_string);

    if ($date->hasErrors()) {
      return NULL;
    }

    return $date;
  }
}
