<?php

namespace Drupal\drw\Plugin\Field\FieldWidget;

use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\datetime\Plugin\Field\FieldWidget\DateTimeWidgetBase;
use Drupal\drw\DateParserTrait;

/**
 * Plugin implementation of the 'drw_date_range' widget.
 *
 * @FieldWidget(
 *   id = "drw_date_range",
 *   label = @Translation("Date with range validation"),
 *   field_types = {
 *     "datetime"
 *   }
 * )
 */
class DateRangeWidget extends DateTimeWidgetBase
{

  use DateParserTrait;

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings(): array {
 
    return [
      'min_date' => '',
      'max_date' => '',
      'min_error_message' => '',
      'max_error_message' => '',
      'required_error_message' => '',
    ] + parent::defaultSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state): array {
 
    $element = parent::settingsForm($form, $form_state);

    $element['min_date'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Minimum date'),
      '#default_value' => $this->getSetting('min_date'),
      '#description' => $this->t('Enter a date or relative format. Examples: "today", "-18 years", "2000-01-01", "-2 weeks". Leave empty for no minimum.'),
    ];

    $element['max_date'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Maximum date'),
      '#default_value' => $this->getSetting('max_date'),
      '#description' => $this->t('Enter a date or relative format. Examples: "today", "+1 year", "2025-12-31", "+3 months". Leave empty for no maximum.'),
    ];

    $element['min_error_message'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Minimum date error message'),
      '#default_value' => $this->getSetting('min_error_message'),
      '#placeholder' => $this->t('The date must be on or after @min.'),
      '#description' => $this->t('Error message when date is before minimum. Use @min as placeholder for the minimum date.'),
    ];

    $element['max_error_message'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Maximum date error message'),
      '#default_value' => $this->getSetting('max_error_message'),
      '#placeholder' => $this->t('The date must be on or before @max.'),
      '#description' => $this->t('Error message when date is after maximum. Use @max as placeholder for the maximum date.'),
    ];

    $element['required_error_message'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Required field error message'),
      '#default_value' => $this->getSetting('required_error_message'),
      '#description' => $this->t('Custom error message when field is required but empty. Leave empty to use default Drupal message.'),
    ];

    return $element;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary(): array {
 
    $summary = parent::settingsSummary();

    $min_date = $this->getSetting('min_date');
    $max_date = $this->getSetting('max_date');

    if (!empty($min_date)) {
      $summary[] = $this->t('Minimum date: @min', ['@min' => $min_date]);
    }

    if (!empty($max_date)) {
      $summary[] = $this->t('Maximum date: @max', ['@max' => $max_date]);
    }

    if (empty($min_date) && empty($max_date)) {
      $summary[] = $this->t('No date range restrictions');
    }

    return $summary;
  }

  /**
   * {@inheritdoc}
   */
  public function formElement(FieldItemListInterface $items, $delta, array $element, array &$form, FormStateInterface $form_state): array {
 
    $element = parent::formElement($items, $delta, $element, $form, $form_state);

    // Pass the title to the value sub-element so it reaches datetime_wrapper template.
    if (!empty($element['#title']) && isset($element['value'])) {
      $element['value']['#title'] = $element['#title'];
    }

    // Force date-only format (no time or seconds).
    $element['value']['#date_time_element'] = 'none';
    $element['value']['#date_time_format'] = '';

    // Add custom required error message if configured.
    $required_message = $this->getSetting('required_error_message');
    if (!empty($required_message) && !empty($element['#required'])) {
      $element['value']['#required_error'] = $required_message;
    }

    // Add data attribute with field name for identification.
    $field_name = $items->getName();
    $element['#attributes']['data-date-range-field'] = $field_name;
    $element['value']['#attributes']['data-date-range-field'] = $field_name;

    // Add HTML5 min and max attributes.
    $min_date = $this->getSetting('min_date');
    $max_date = $this->getSetting('max_date');

    if (!empty($min_date)) {
      $min_datetime = $this->parseRelativeDate($min_date);

      if ($min_datetime) {
        $element['value']['#attributes']['min'] = $min_datetime->format('Y-m-d');
      }
    }

    if (!empty($max_date)) {
      $max_datetime = $this->parseRelativeDate($max_date);
      if ($max_datetime) {
        $element['value']['#attributes']['max'] = $max_datetime->format('Y-m-d');
      }
    }

    return $element;
  }

}
