<?php

namespace Drupal\drw\Plugin\Validation\Constraint;

use Drupal\Core\Datetime\DrupalDateTime;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\drw\DateParserTrait;
use Symfony\Component\DependencyInjection\Attribute\Autowire;
use Symfony\Component\Validator\Constraint;
use Symfony\Component\Validator\ConstraintValidator;

/**
 * Validates the DateRange constraint.
 */
class DateRangeConstraintValidator extends ConstraintValidator {

  use DateParserTrait;

  /**
   * Constructs a DateRangeConstraintValidator object.
   *
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory.
   */
  public function __construct(
    #[Autowire(service: 'logger.factory')]
    protected LoggerChannelFactoryInterface $loggerFactory,
  ) {
  }

  /**
   * {@inheritdoc}
   *
   * @param mixed $item
   *   The item to validate.
   * @param \Drupal\drw\Plugin\Validation\Constraint\DateRangeConstraint $constraint
   *   The constraint.
   */
  public function validate($item, Constraint $constraint): void {
    if (!isset($item)) {
      return;
    }

    $value = $item->value;
    if (empty($value)) {
      return;
    }

    if (is_array($value)) {
      return;
    }

    try {
      // Parse submitted date (normalize to date only, no time component).
      $submitted_date = new DrupalDateTime($value);

      if ($submitted_date->hasErrors()) {
        return;
      }

      // Normalize to date only (start of day) for comparison.
      $submitted_date->setTime(0, 0, 0);

      // Validate minimum date.
      if (!empty($constraint->minDate)) {
        $min_date = $this->parseRelativeDate($constraint->minDate);
        if ($min_date) {
          $min_date->setTime(0, 0, 0);
          if ($submitted_date < $min_date) {
            $message = str_replace('@min', $min_date->format('Y-m-d'), $constraint->minErrorMessage);
            $this->context->addViolation($message);
            return;
          }
        }
      }

      // Validate maximum date.
      if (!empty($constraint->maxDate)) {
        $max_date = $this->parseRelativeDate($constraint->maxDate);
        if ($max_date) {
          $max_date->setTime(0, 0, 0);
          if ($submitted_date > $max_date) {
            $message = str_replace('@max', $max_date->format('Y-m-d'), $constraint->maxErrorMessage);
            $this->context->addViolation($message);
            return;
          }
        }
      }
    }
    catch (\Exception $e) {
      $this->loggerFactory->get('drw')->error('Error in DateRange validation: @error', ['@error' => $e->getMessage()]);
    }
  }

}
