<?php

namespace Drupal\drw\Plugin\Field\FieldWidget;

use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\datetime\Plugin\Field\FieldWidget\DateTimeWidgetBase;
use Drupal\drw\DateParserTrait;

/**
 * Plugin implementation of the 'drw_date_range' widget.
 *
 * @FieldWidget(
 *   id = "drw_date_range",
 *   label = @Translation("Date Range Widget"),
 *   field_types = {
 *     "datetime"
 *   }
 * )
 */
class DateRangeWidget extends DateTimeWidgetBase
{

  use DateParserTrait;

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings(): array {
 
    return [
      'min_date' => '',
      'max_date' => '',
      'min_error_message' => '',
      'max_error_message' => '',
      'required_error_message' => '',
      'enable_custom_error_messages' => FALSE,
    ] + parent::defaultSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state): array {
 
    $element = parent::settingsForm($form, $form_state);

    $element['min_date'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Minimum date'),
      '#default_value' => $this->getSetting('min_date'),
      '#description' => $this->t('Enter a date or relative format. Examples: "today", "-18 years", "2000-01-01", "-2 weeks". Leave empty for no minimum.'),
    ];

    $element['max_date'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Maximum date'),
      '#default_value' => $this->getSetting('max_date'),
      '#description' => $this->t('Enter a date or relative format. Examples: "today", "+1 year", "2025-12-31", "+3 months". Leave empty for no maximum.'),
    ];

    $element['enable_custom_error_messages'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable custom error messages'),
      '#default_value' => $this->getSetting('enable_custom_error_messages'),
      '#description' => $this->t('Allows you to define custom error messages for this field. <strong>Warning:</strong> This disables HTML5 validation for the entire form and relies on server-side validation for required fields.'),
    ];

    // Conditional visibility for custom error messages.
    $conditional_visibility = [
      '#states' => [
        'visible' => [
          ':input[name="fields[' . $this->fieldDefinition->getName() . '][settings_edit_form][settings][enable_custom_error_messages]"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $element['min_error_message'] = $conditional_visibility + [
      '#type' => 'textfield',
      '#title' => $this->t('Error message for minimum date'),
      '#default_value' => $this->getSetting('min_error_message'),
      '#placeholder' => $this->t('The date must be on or after @min.'),
      '#description' => $this->t('Error message when date is before minimum. Use @min as placeholder for the minimum date.'),
    ];

    $element['max_error_message'] = $conditional_visibility + [
      '#type' => 'textfield',
      '#title' => $this->t('Error message for maximum date'),
      '#default_value' => $this->getSetting('max_error_message'),
      '#placeholder' => $this->t('The date must be on or before @max.'),
      '#description' => $this->t('Error message when date is after maximum. Use @max as placeholder for the maximum date.'),
    ];

    $element['required_error_message'] = $conditional_visibility + [
      '#type' => 'textfield',
      '#title' => $this->t('Error message for required flag'),
      '#default_value' => $this->getSetting('required_error_message'),
      '#description' => $this->t('Custom error message when field is required but empty. Leave empty to use default Drupal message.'),
    ];

    return $element;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary(): array {
 
    $summary = parent::settingsSummary();

    $min_date = $this->getSetting('min_date');
    $max_date = $this->getSetting('max_date');

    if (!empty($min_date)) {
      $summary[] = $this->t('Minimum date: @min', ['@min' => $min_date]);
    }

    if (!empty($max_date)) {
      $summary[] = $this->t('Maximum date: @max', ['@max' => $max_date]);
    }

    if (empty($min_date) && empty($max_date)) {
      $summary[] = $this->t('No date range restrictions');
    }

    return $summary;
  }

  /**
   * {@inheritdoc}
   */
  public function formElement(FieldItemListInterface $items, $delta, array $element, array &$form, FormStateInterface $form_state): array {
 
    $element = parent::formElement($items, $delta, $element, $form, $form_state);

    // Pass the title to the value sub-element so it reaches datetime_wrapper template.
    if (!empty($element['#title']) && isset($element['value'])) {
      $element['value']['#title'] = $element['#title'];
    }

    // Force date-only format (no time or seconds).
    $element['value']['#date_time_element'] = 'none';
    $element['value']['#date_time_format'] = '';

    // Add custom required error message if configured.
    $required_message = $this->getSetting('required_error_message');
    if (!empty($required_message) && !empty($element['#required'])) {
      $element['value']['#required_error'] = $this->t($required_message);
    }

    // Add data attribute with field name for identification.
    $field_name = $items->getName();
    $element['#attributes']['data-date-range-field'] = $field_name;
    $element['value']['#attributes']['data-date-range-field'] = $field_name;

    // Add HTML5 min and max attributes.
    foreach (['min' => 'min_date', 'max' => 'max_date'] as $attr => $setting) {
      $date_value = $this->getSetting($setting);
      if (!empty($date_value)) {
        $datetime = $this->parseRelativeDate($date_value);
        if ($datetime) {
          $element['value']['#attributes'][$attr] = $datetime->format('Y-m-d');
        }
      }
    }

    // Disable HTML5 validation to allow custom error messages.
    if ($this->getSetting('enable_custom_error_messages')) {
      $form['#attributes']['novalidate'] = 'novalidate';
    }

    return $element;
  }

}
