/**
 * @file
 * Manage tabbledrag features with DSFR.
 */

/**
 * Triggers when weights columns are toggled.
 *
 * @event columnschange
 */

(function ($, Drupal) {
  "use strict";

  const initColumnsOriginal = Drupal.tableDrag.prototype.initColumns;
  const addChangedWarningOriginal = Drupal.tableDrag.prototype.row.prototype.addChangedWarning;

  /**
   * {@inheritdoc}
   */
  Drupal.tableDrag.prototype.row.prototype.addChangedWarning = function () {
    const $table = $(this.table.parentNode);

    // Do not add the changed warning if one is already present.
    if (!$table.find(".tabledrag-changed-warning").length) {
      addChangedWarningOriginal.call(this);

      const $tableWrapper = $table.parents(".fr-table");

      // Move changed warning message before DSFR table wrapper.
      $tableWrapper.parent().prepend($table.find(".tabledrag-changed-warning"));
    }
  };

  /**
   * {@inheritdoc}
   */
  Drupal.tableDrag.prototype.initColumns = function () {
    const $tableWrapper = this.$table.parents(".fr-table");
    const $toggleWeightWrapper = this.$table.prev();

    // Move toggle weight wrapper before DSFR table wrapper.
    $tableWrapper.before($toggleWeightWrapper);

    // Call original "initColumns" method.
    initColumnsOriginal.call(this);
  };

  $.extend(
    Drupal.theme,
    /** @lends Drupal.theme */ {
      /**
       * Constructs contents of the toggle weight button.
       *
       * @param {boolean} show
       *   If the table weights are currently displayed.
       *
       * @return {string}
       *  HTML markup for the weight toggle button content.
       */
      toggleButtonContent: (show) => {
        const classes = [
          "tabledrag-toggle-weight",
          "fr-icon--sm",
        ];
        let text = "";
        if (show) {
          classes.push("fr-icon-eye-off-line");
          text = Drupal.t("Hide row weights");
        }
 else {
          classes.push("fr-icon-eye-line");
          text = Drupal.t("Show row weights");
        }
        return `<span class="${classes.join(" ")}" aria-hidden="true"></span>&nbsp;${text}`;
      },
    },
  );

})(jQuery, Drupal);
