<?php

declare(strict_types=1);

namespace Drupal\dsfr4drupal_colors\Plugin\Field\FieldFormatter;

use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Field\Attribute\FieldFormatter;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\Utility\Token;
use Drupal\token\TokenEntityMapperInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation of the dsfr4drupal_color field CSS declaration formatter.
 */
#[FieldFormatter(
  id: 'dsfr4drupal_color_field_formatter_css',
  label: new TranslatableMarkup('Color CSS declaration'),
  field_types: [
    'dsfr4drupal_color_field_type',
  ],
)]
class ColorFieldFormatterCss extends ColorFieldFormatterBase implements ContainerFactoryPluginInterface {

  /**
   * The module handler.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected ModuleHandlerInterface $moduleHandler;

  /**
   * The token service.
   *
   * @var \Drupal\Core\Utility\Token
   */
  protected Token $tokenService;

  /**
   * The token entity mapper service.
   *
   * @var \Drupal\token\TokenEntityMapperInterface|null
   */
  protected ?TokenEntityMapperInterface $tokenEntityMapper = NULL;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);

    $instance->moduleHandler = $container->get('module_handler');
    $instance->tokenService = $container->get('token');
    $instance->tokenEntityMapper = $container->has('token.entity_mapper') ? $container->get('token.entity_mapper') : NULL;

    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings(): array {
    return [
        'selector' => 'body',
        'property' => 'background-color',
        'important' => TRUE,
        'advanced' => FALSE,
        'css' => '',
      ] + parent::defaultSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state): array {
    $elements = [];

    $elements['selector'] = [
      '#title' => $this->t('Selector'),
      '#description' => $this->t('A valid CSS selector such as <code>.links > li > a, #logo</code>. You can use tokens as shown below.'),
      '#type' => 'textarea',
      '#rows' => '1',
      '#default_value' => $this->getSetting('selector'),
      '#required' => TRUE,
      '#placeholder' => 'body > div > a',
      '#states' => [
        'visible' => [
          ':input[name="fields[' . $this->fieldDefinition->getName() . '][settings_edit_form][settings][advanced]"]' => ['checked' => FALSE],
        ],
      ],
    ];
    $elements['property'] = [
      '#title' => $this->t('Property'),
      '#type' => 'select',
      '#default_value' => $this->getSetting('property'),
      '#required' => TRUE,
      '#options' => [
        'background-color' => $this->t('Background color'),
        'color' => $this->t('Text color'),
      ],
      '#states' => [
        'visible' => [
          ':input[name="fields[' . $this->fieldDefinition->getName() . '][settings_edit_form][settings][advanced]"]' => ['checked' => FALSE],
        ],
      ],
    ];
    $elements['important'] = [
      '#title' => $this->t('Important'),
      '#description' => $this->t('To make this statement more important than others.'),
      '#type' => 'checkbox',
      '#default_value' => $this->getSetting('important'),
      '#states' => [
        'visible' => [
          ':input[name="fields[' . $this->fieldDefinition->getName() . '][settings_edit_form][settings][advanced]"]' => ['checked' => FALSE],
        ],
      ],
    ];

    $elements['advanced'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Advanced mode'),
      '#default_value' => $this->getSetting('advanced'),
      '#description' => t('Switch to advanced mode and build the css yourself.'),
    ];

    $elements['css'] = [
      '#type' => 'textarea',
      '#title' => $this->t('CSS'),
      '#default_value' => $this->getSetting('css'),
      '#description' => t('Create the css statement yourself. This lets you for example, control multiple element aspects at once. You can use tokens as shown below.'),
      '#states' => [
        'visible' => [
          ':input[name="fields[' . $this->fieldDefinition->getName() . '][settings_edit_form][settings][advanced]"]' => ['checked' => TRUE],
        ],
      ],
    ];
    if ($this->moduleHandler->moduleExists('token')) {
      $elements['css']['#element_validate'][] = 'token_element_validate';
      $elements['css']['#token_types'] = [
        $this->getTokenType(),
        'dsfr4drupal_color_field',
      ];

      $elements['token_help'] = [
        '#theme' => 'token_tree_link',
        '#token_types' => [
          $this->getTokenType(),
          'dsfr4drupal_color_field',
        ],
      ];
    }

    return $elements;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary(): array {
    $settings = $this->getSettings();

    $summary = [];

    if ($settings['advanced']) {
      $summary[] = $this->t('Using advanced mode');
      $summary[] = $this->t("CSS statement:\n@css", ['@css' => $settings['css']]);
    }
    else {
      $summary[] = $this->t('CSS selector: @css_selector', [
        '@css_selector' => $settings['selector'],
      ]);
      $summary[] = $this->t('CSS property: @css_property', [
        '@css_property' => $settings['property'],
      ]);
      $summary[] = $this->t('!important declaration: @important_declaration', [
        '@important_declaration' => ($settings['important'] ? $this->t('Yes') : $this->t('No')),
      ]);
    }

    return $summary;
  }

  /**
   * {@inheritdoc}
   */
  public function viewElements(FieldItemListInterface $items, $langcode): array {
    $settings = $this->getSettings();

    $elements = [];

    $tokens = [
      $this->getTokenType() => $items->getEntity(),
    ];

    foreach ($items as $item) {
      $tokens['dsfr4drupal_color_field'] = $item;

      if ($settings['advanced']) {
        $inlineCSS = $this->tokenService->replace(
          $settings['css'],
          $tokens
        );
      }
      else {
        $selector = $this->tokenService->replace(
          $settings['selector'],
          $tokens
        );
        $inlineCSS = $selector . ' { ';
        $inlineCSS .= $settings['property'] . ': ';
        $inlineCSS .= $this->helper->getColorCssValue($this->viewRawValue($item));
        $inlineCSS .= ($settings['important'] ? ' !important' : '') . ';';
        $inlineCSS .= ' }';
      }

      $elements['#attached']['html_head'][] = [
        [
          '#tag' => 'style',
          '#value' => $inlineCSS,
        ],
        sha1($inlineCSS),
      ];
      // If rendered in a view entity field, the #attached only propagates if
      // there is some markup set.
      $elements[0] = [
        '#type' => 'html_tag',
        '#tag' => 'div',
        '#attributes' => ['class' => 'hidden'],
        '#value' => $this->viewRawValue($item),
      ];
    }

    return $elements;
  }

  /**
   * Gets the token type of the target entity.
   *
   * If the token entity mapper service is available, it will be used to get
   * the token type. If that service is not available, the target entity type id
   * will be used as a fallback.
   *
   * @return string
   *   Token type of the target entity.
   */
  protected function getTokenType(): string {
    $entity_type_id = $this->fieldDefinition->getTargetEntityTypeId();

    if (!$this->tokenEntityMapper) {
      return $entity_type_id;
    }

    return $this->tokenEntityMapper->getTokenTypeForEntityType($entity_type_id, TRUE);
  }

}
