/**
 * @file
 * Attaches behaviors for DSFR color field box widget.
 */

((Drupal, once) => {
  "use strict";

  /**
   * Convert hex color value to rgb color value.
   *
   * @param {string} hex
   *   A hexadecimal color.
   *
   * @return {{r: number, b: number, g: number}|null}
   *   An object containing the RGB value.
   */
  const hexToRgb = (hex) => {

    // Expand shorthand form (e.g. '03F') to full form (e.g. '0033FF')
    const shorthandRegex = /^#?([a-f\d])([a-f\d])([a-f\d])$/i;

    hex = hex.replace(shorthandRegex, function(m, r, g, b) {
      return r + r + g + g + b + b;
    });

    const result = /^#?([a-f\d]{2})([a-f\d]{2})([a-f\d]{2})$/i.exec(hex);

    // Returns {r: '', g: '', b: ''}
    return result ? {
      r: parseInt(result[1], 16),
      g: parseInt(result[2], 16),
      b: parseInt(result[3], 16)
    } : null;
  };

  /**
   * Calculate color contrast with RGB values.
   *
   * @param {object} rgb1
   *   A first object containing an RGB value.
   * @param {object} rgb2
   *   A second object containing an RGB value.
   *
   * @return {number}
   *   The contrast ratio.
   */
  const contrast = (rgb1, rgb2) => {
    return Math.max(rgb1.r, rgb2.r) - Math.min(rgb1.r, rgb2.r) +
      Math.max(rgb1.g, rgb2.g) - Math.min(rgb1.g, rgb2.g) +
      Math.max(rgb1.b, rgb2.b) - Math.min(rgb1.b, rgb2.b);
  };

  Drupal.behaviors.dsfr4drupalColorContrastRatio = {
    attach: (context, settings) => {
      const settingsColors = settings.dsfr4drupal_colors;
      const settingsContrastRatio = settingsColors.contrastRatio || {};

      Object.entries(settingsContrastRatio).forEach(([instanceId, instanceSettings]) => {
        const wrapper = context.getElementById(instanceId);
        if (!wrapper) {
          return;
        }

        const input = wrapper.querySelector("input");

        input.addEventListener("change", () => {
          // Reset input validity.
          input.setCustomValidity("");

          let value2 = instanceSettings.value;

          if (instanceSettings.type === "name" && value2) {
            value2 = settingsColors.colorsAvailable[value2];
          }
          else if (instanceSettings.type === "field") {
            value2 = document.getElementById(instanceSettings.value).value;

            if (value2 !== "") {
              value2 = settingsColors.colorsAvailable[value2];
            }
          }

          // Check all fields are defined.
          if (input.value && value2) {
            const contrastColor = contrast(
              hexToRgb(settingsColors.colorsAvailable[input.value]),
              hexToRgb(value2),
            );

            if (contrastColor <= 450) {
              // Throw input error.
              input.setCustomValidity(
                Drupal.t(
                  "The contrast ratio is @ratio. The minimum required ratio in accessibility is 450.",
                  { "@ratio": contrastColor }
                )
              );
              input.reportValidity();
            }
          }
        });
      });
    },
  };

})(Drupal, once);
