<?php

declare(strict_types=1);

namespace Drupal\dsfr4drupal_colors\Helper;

use Drupal\Core\Cache\UseCacheBackendTrait;
use Drupal\Core\File\FileExists;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;

/**
 * Provides a ColorsHelper class.
 */
class ColorsHelper implements ColorsHelperInterface {

  use UseCacheBackendTrait;
  use StringTranslationTrait;

  /**
   * Creates a ColorsHelper instance.
   *
   * @param \Drupal\Core\File\FileSystemInterface $fileSystem
   *     The file system service.
   */
  final public function __construct(
    protected FileSystemInterface $fileSystem
  ) {
  }

  /**
   * Find color declarations into given content.
   *
   * @param string $content
   *   The content to search.
   *
   * @return array
   *   The color declarations.
   */
  private function findColors(string $content): array {
    $colors = [];

    // Looking for color declarations (without "active" and "hover").
    if (preg_match_all(
      '@^\s*--((?!.+-(?:hover|active)).+):\s*(#[a-f0-9]+)\s*;\s*$@im',
      $content,
      $matches
    )) {
      foreach ($matches[1] as $index => $match) {
        $colors[$matches[2][$index]] = $match;
      }
    }

    return $colors;
  }

  /**
   * {@inheritdoc}
   */
  public function generateCssColorsFile(): void {
    $colors = $this->parseCoreFile();
    $colorsDark = $this->parseSchemeFile();

    $data = ":root,\n";
    $data .= '.' . static::CLASS_THEME_LIGHT . " {\n";
    foreach ($colors as $hex => $name) {
      $data .= "  --{$name}: {$hex};\n";
    }
    $data .= "}\n\n";

    $data .= ":root[data-fr-theme=dark],\n";
    $data .= '.' . static::CLASS_THEME_DARK . " {\n";
    foreach ($colorsDark as $hex => $name) {
      $data .= "  --{$name}: {$hex};\n";
    }
    $data .= "}\n";

    $this->fileSystem->saveData(
      $data,
      'public://dsfr4drupal-colors.css',
      FileExists::Replace
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getColorCssValue(string $color): string {
    return 'var(' . $this->getColorCssVariable($color) . ')';
  }

  /**
   * {@inheritdoc}
   */
  public function getColorCssVariable(string $color): string {
    return '--' . $color;
  }

  /**
   * {@inheritdoc}
   */
  public function getColorsAvailable(): array {
    $cid = 'dsfr4drupal_colors:colors';
    $cache = $this->cacheGet($cid);
    if ($cache) {
      return $cache->data;
    }

    $colors = $this->parseCoreFile();
    $this->sortColors($colors);

    $this->cacheSet($cid, $colors);

    return $colors;
  }

  /**
   * {@inheritdoc}
   */
  public function getColorsOptions(): array {
    $colors = $this->getColorsAvailable();

    return array_combine($colors, $colors);
  }

  /**
   * {@inheritdoc}
   */

  public function getColorsOptionsByGroups(): array {
    $colorsOptions = $this->getColorsOptions();
    $groups = $this->getGroups();

    $options = [];
    foreach ($colorsOptions as $color => $label) {
      $group = 'illustrative';
      foreach ($groups as $groupKey => $groupData) {
        foreach ($groupData['group_colors'] as $groupColor) {
          if (str_starts_with($color, $groupColor . '-')) {
            $group = $groupKey;
            break 2;
          }
        }
      }

      $options[$group][$color] = $label;
    }

    return $options;
  }

  /**
   * {@inheritdoc}
   */
  public function getCssCorePath(): string {
    return self::LIBRARY_PATH . 'core/core.css';
  }

  /**
   * {@inheritdoc}
   */
  public function getCssSchemePath(): string {
    return self::LIBRARY_PATH . 'scheme/scheme.css';
  }

  /**
   * {@inheritdoc}
   */
  public function getGroups(): array {
    return [
      'primary' => [
        'label' => $this->t('Primary colors'),
        'description' => $this->t(
          'Built from the two main colors of the State brand (blue and red, with white integrated into the neutrals), they are used to mark the identity of the State in components that convey the brand image (such as the brand block), or on which it is necessary to attract the user\'s attention, such as clickable elements or active states.'
        ),
        'group_colors' => [
          'blue-france',
          'red-marianne',
        ],
      ],
      'neutral' => [
        'label' => $this->t('Neutral colors'),
        'description' => $this->t(
          'Basic colors used in typography, backgrounds, outlines, and separators in most components. They are notably used in non-clickable elements and to represent inactive states.'
        ),
        'group_colors' => [
          'grey',
        ],
      ],
      'system' => [
        'label' => $this->t('System colors'),
        'description' => $this->t('Colors used exclusively to represent states and statuses.'),
        'group_colors' => [
          'info',
          'success',
          'warning',
          'error',
        ],
      ],
      'illustrative' => [
        'label' => $this->t('Illustrative colors'),
        'description' => $this->t(
          'Complementary colors from the State\'s charter, which can be used for illustrative composition. Within the context of the State\'s Design System, these colors can also be used to accentuate components, i.e., vary the color of certain elements (text, backgrounds, borders) to provide diversity or visual hierarchy. <br /><br />The use of illustrative colors is possible on certain components: <br />See the list of accentuable components.',

        ),
        'group_colors' => [
          // Other colors.
        ],
      ],
    ];
  }

  /**
   * Parse CSS core file to find color declarations.
   *
   * @return array
   *   An associative array containing the color declaration.
   *   THe key is the hex and the value the variable.
   */
  private function parseCoreFile(): array {
    $content = file_get_contents(
      $this->getCssCorePath()
    );

    // Get ":root { ... }" declaration.
    if (preg_match(
      '#:root\s*{(.*?)}#is',
      $content,
      $matches
    )) {;
      return $this->findColors($matches[1]);
    }

    return [];
  }

  /**
   * Parse CSS scheme file to find color declarations of the dark theme.
   *
   * @return array
   *   An associative array containing the color declaration.
   *   THe key is the hex and the value the variable.
   */
  private function parseSchemeFile(): array {
    $content = file_get_contents(
      $this->getCssSchemePath()
    );

    // Get ":root { ... }" declaration.
    if (preg_match(
      '#:root\[data-fr-theme=(?:"|\')?dark(?:"|\')?\]\s*{(.*?)}#is',
      $content,
      $matches
    )) {;
      return $this->findColors($matches[1]);
    }

    return [];
  }

  /**
   * Sort colors.
   * @param array $colors
   *   The available colors.
   */
  private function sortColors(array &$colors): void {
    /**
     * Token convention: color-name-variant-hint-state.
     * @see: https://www.systeme-de-design.gouv.fr/fondamentaux/couleurs-palette/
     */

    // Define a "none" value for a missing value.
    // For example, system color tokens (info, success, warning, error) haven't a name.
    $none = '__NONE__';

    // Group colors by name prefix (= primary color) and.
    $colorsGrouped = [];
    foreach ($colors as $hex => $name) {
      // Group colors by name prefix (= primary color) and declination.
      preg_match('#^([a-z]+)(?:-([a-z-]+(?<!main|sun)))?-(main|sun|[0-9]+)?-(.*)$#', $name, $match);
      [, $color, $colorName, $variant, $hint] = $match;
      $colorsGrouped[$color][$colorName ?: $none][$variant][$hint] = $hex;
    }

    // Sort colors group by name (unknown colors are displayed first).
    foreach (static::COLOR_ORDER as $color) {
      if (isset($colorsGrouped[$color])) {
        $colorColors = $colorsGrouped[$color];
        unset($colorsGrouped[$color]);
        $colorsGrouped[$color] = $colorColors;
      }
    }

    foreach (array_keys($colorsGrouped) as $color) {
      // There is no point in sorting the declensions by name, that would be purely arbitrary.

      // Sort colors into a declination.
      foreach ($colorsGrouped[$color] as $colorName => &$colorVariants) {
        $this->sortColorVariants($color, $colorName, $colorVariants);
      }
    }

    // Flat multidimensional array.
    $colors = [];
    foreach ($colorsGrouped as $color => $colorNames) {
      foreach ($colorNames as $colorName => $variants) {
        foreach($variants as $variant => $hints) {
          foreach ($hints as $hint => $hex) {
            if ($colorName === $none) {
              $name = sprintf('%s-%s-%s', $color, $variant, $hint);
            }
            else {
              $name = sprintf('%s-%s-%s-%s', $color, $colorName, $variant, $hint);
            }
          }

          $colors[$hex] = $name;
        }
      }
    }
  }

  /**
   * Sort colors variants for a given color name.
   *
   * @param string $color
   *   The color.
   * @param string $colorName
   *   The color name.
   * @param array $variants
   *   The color variants.
   */
  private function sortColorVariants(string $color, string $colorName, array &$variants): void {
    krsort($variants);

    // Set the color "main" variant last.
    if (isset($variants['main'])) {
      $hints = $variants['main'];
      unset($variants['main']);
      $variants = $variants + ['main' => $hints];
    }

    // Set the color "sun" or "moon"" variant last (and after "main").
    if (isset($variants['moon'])) {
      $hints = $variants['moon'];
      unset($variants['moon']);
      $variants = $variants + ['moon' => $hints];
    }
    if (isset($variants['sun'])) {
      $hints = $variants['sun'];
      unset($variants['sun']);
      $variants = $variants + ['sun' => $hints];
    }

    // Sort variant hints.
    foreach (array_keys($variants) as $variant) {
      krsort($variants[$variant]);
    }
  }

}
