<?php

declare(strict_types=1);

namespace Drupal\dsfr4drupal_colors\Plugin\Field\FieldType;

use Drupal\Core\Field\Attribute\FieldType;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Field\FieldItemBase;
use Drupal\Core\Field\FieldStorageDefinitionInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Form\OptGroup;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\TypedData\DataDefinition;

/**
 * Plugin implementation of the 'dsfr4drupal_color_field_type' field type.
 */
#[FieldType(
  id: 'dsfr4drupal_color_field_type',
  label: new TranslatableMarkup('DSFR for Drupal - Color'),
  description: new TranslatableMarkup('Create and store color value.'),
  default_widget: 'dsfr4drupal_color_field_widget_box',
  default_formatter: 'dsfr4drupal_color_field_formatter_text',
  constraints: [
    'ComplexData' => [
      'color' => [
        'Regex' => [
          'pattern' => '#^[a-z0-9-]+-[0-9]+$#',
        ],
      ],
    ]
  ]
)]
class ColorFieldType extends FieldItemBase {

  /**
   * Hex value of color.
   *
   * @var string
   */
  protected string $color;

  /**
   * {@inheritdoc}
   */
  public static function defaultFieldSettings(): array {
    return parent::defaultFieldSettings() + [
        'allowed_colors' => [],
        'contrast_ratio_type' => '',
        'contrast_ratio_value' => '',
      ];
  }

  /**
   * {@inheritdoc}
   */
  public function fieldSettingsForm(array $form, FormStateInterface $form_state): array {
    $form = parent::fieldSettingsForm($form, $form_state);

    $this->fieldSettingsFormAllowedColors($form, $form_state);
    $this->fieldSettingsFormContrastRatio($form, $form_state);

    return $form;
  }

  /**
   * Returns an "allowed_colors" element for the field-level settings.
   *
   * @param array $form
   *   The form where the settings form is being included in.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state of the (entire) configuration form.
   */
  private function fieldSettingsFormAllowedColors(array &$form, FormStateInterface $form_state): void {
    /** @var \Drupal\dsfr4drupal_colors\Helper\ColorsHelperInterface $helper */
    $helper = \Drupal::service('dsfr4drupal_colors.helper.colors');
    $options = $helper->getColorsOptionsByGroups();
    $allowedColors = $this->getSetting('allowed_colors') ?? [];

    $form['allowed_colors'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Allowed colors'),
      '#element_validate' => [[static::class, 'validateAllowedColors']],
    ];
    $form['allowed_colors']['description'] = [
      '#type' => 'item',
      '#markup' => $this->t('Select the colors that can be picked. If no colors are checked, all colors will be available.'),
    ];
    foreach ($helper->getGroups() as $group => $data) {
      $form['allowed_colors'][$group] = [
        '#type' => 'details',
        '#title' => $data['label'],
        '#description' => $data['description'],
        '#open' => FALSE,
        '#attributes' => [
          'class' => [
            'dsfr4drupal-colors-allowed-colors-group',
            'dsfr4drupal-colors-allowed-colors-group--' . $group,
          ],
        ],
        '#attached' => [
          'library' => ['dsfr4drupal_colors/allowed-colors-select'],
        ],
      ];

      foreach ($options[$group] as $name => $label) {
        $form['allowed_colors'][$group][$name] = [
          '#type' => 'checkbox',
          '#title' => $label,
          '#return_value' => $name,
          '#parents' => array_merge($element['#parents'] ?? ['settings'], ['allowed_colors', $name]),
          '#default_value' => in_array($name, $allowedColors),
          // Errors should only be shown on the parent element.
          '#error_no_message' => TRUE,
        ];
      }
    }

    // Now that we've processed, collapse the values so that form validator
    // doesn't flag the selection as invalid.
    $form['allowed_colors']['#options'] = OptGroup::flattenOptions($options);
  }

  /**
   * Returns a "contrast_ratio" element for the field-level settings.
   *
   * @param array $form
   *   The form where the settings form is being included in.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state of the (entire) configuration form.
   */
  private function fieldSettingsFormContrastRatio(array &$form, FormStateInterface $form_state): void {
    $form['contrast_ratio'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Contrast ratio validation'),
      '#element_validate' => [[static::class, 'validateContrastRatio']],
    ];
    $form['contrast_ratio']['description'] = [
      '#type' => 'item',
      '#markup' => $this->t(
        'The minimum required accessibility ratio between two colors is <strong>450</strong>.<br />You can validate the contrast of the color selected in this field by configuring the fields below.'
      ),
    ];

    $form['contrast_ratio']['type'] = [
      '#type' => 'select',
      '#title' => $this->t('Validation type'),
      '#options' => [
        'field' => $this->t('Another "DSFR for Drupal - Color" field type'),
        'hex' => $this->t('A hexadecimal color code'),
        'name' => $this->t('A DSFR color variable name'),
      ],
      '#empty_option' => $this->t('- No validation -'),
      '#default_value' => $this->getSetting('contrast_ratio_type') ?? '',
    ];

    /** @var \Drupal\Core\Field\FieldDefinitionInterface[] $fields */
    $fields = \Drupal::service('entity_field.manager')->getFieldDefinitions(
      $this->getEntity()->getEntityTypeId(),
      $this->getEntity()->bundle()
    );

    $options = [];
    foreach ($fields as $field) {
      if (
        $field->getType() === 'dsfr4drupal_color_field_type' &&
        $field->getName() !== $this->getFieldDefinition()->getName()
      ) {
        $options[$field->getName()] = $field->getLabel();
      }
    }

    if (empty($options)) {
      $form['contrast_ratio']['field'] = [
        '#type' => 'container',
        'description' => [
          '#type' => 'item',
          '#markup' => $this->t(
            'No other fields were found.<br />You must create at least two fields of type "DSFR for Drupal - Color" to use this feature.'
          ),
        ],
        '#states' => [
          'visible' => [
            ':input[name="settings[contrast_ratio][type]"]' => ['value' => 'field'],
          ],
        ],
      ];
    }
    else {
      $form['contrast_ratio']['field'] = [
        '#type' => 'select',
        '#title' => $this->t('Available fields'),
        '#options' => $options,
        '#states' => [
          'visible' => [
            ':input[name="settings[contrast_ratio][type]"]' => ['value' => 'field'],
          ],
          'required' => [
            ':input[name="settings[contrast_ratio][type]"]' => ['value' => 'field'],
          ],
        ],
        '#default_value' => (
          $this->getSetting('contrast_ratio_type') === 'field' &&
          $this->getSetting('contrast_ratio_value')
        ) ? $this->getSetting('contrast_ratio_value') : '',
      ];
    }

    $form['contrast_ratio']['hex'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Hexadecimal color code'),
      '#states' => [
        'visible' => [
          ':input[name="settings[contrast_ratio][type]"]' => ['value' => 'hex'],
        ],
        'required' => [
          ':input[name="settings[contrast_ratio][type]"]' => ['value' => 'hex'],
        ],
      ],
      '#placeholder' => '#fff',
      '#attributes' => [
        // Check is a hex 3 or 6 color code.
        'pattern' => '#([0-9a-fA-F]{3}){1,2}'
      ],
      '#description' => $this->t(
        'You must enter a 3 or 6 character hexadecimal code prefixed by a hash mark "#".'
      ),
      '#default_value' => (
        $this->getSetting('contrast_ratio_type') === 'hex' &&
        $this->getSetting('contrast_ratio_value')
      ) ? $this->getSetting('contrast_ratio_value') : '',
    ];

    $form['contrast_ratio']['name'] = [
      '#type' => 'select',
      '#title' => $this->t('DSFR color variable name'),
      '#options' => \Drupal::service('dsfr4drupal_colors.helper.colors')
        ->getColorsOptions(),
      '#states' => [
        'visible' => [
          ':input[name="settings[contrast_ratio][type]"]' => ['value' => 'name'],
        ],
        'required' => [
          ':input[name="settings[contrast_ratio][type]"]' => ['value' => 'name'],
        ],
      ],
      '#default_value' => (
        $this->getSetting('contrast_ratio_type') === 'name' &&
        $this->getSetting('contrast_ratio_value')
      ) ? $this->getSetting('contrast_ratio_value') : '',
    ];
  }

  /**
   * Render API callback: Processes the allowed colors value.
   *
   * Ensure the element's value is an indexed array of selected color IDs.
   * This function is assigned as an #element_validate callback.
   *
   * @see static::fieldSettingsForm()
   */
  public static function validateAllowedColors(array &$element, FormStateInterface $form_state): void {
    $value = array_values(array_filter($form_state->getValue($element['#parents'])));
    $form_state->setValueForElement($element, $value);
  }

  /**
   * Render API callback: Processes the contrast ratio values.
   *
   * Ensure the element's value is an indexed array of selected color IDs.
   * This function is assigned as an #element_validate callback.
   *
   * @see static::fieldSettingsForm()
   */
  public static function validateContrastRatio(array &$element, FormStateInterface $form_state): void {
    $values = $form_state->getValue($element['#parents']);

    if ($values['type'] === 'field') {
      // In case of any other color field type instance was found.
      if (is_array($values['field'])) {
        $form_state->setError(
          $element['type'],
          t('You cannot select a validation per field.'),
        );
      }
    }

    $form_state->setValue(['settings', 'contrast_ratio_type'], $values['type']);
    $form_state->setValue(['settings', 'contrast_ratio_value'], $values[$values['type']]);
  }

  /**
   * {@inheritdoc}
   */
  public function isEmpty(): bool {
    $value = $this->get('color')->getValue();
    return $value === NULL || $value === '';
  }

  /**
   * {@inheritdoc}
   */
  public static function mainPropertyName(): string {
    return 'color';
  }

  /**
   * {@inheritdoc}
   */
  public static function schema(FieldStorageDefinitionInterface $field_definition): array {
    return [
      'columns' => [
        'color' => [
          'description' => 'The color value',
          'type' => 'varchar',
          'length' => 128,
          'not null' => FALSE,
        ],
      ],
      'indexes' => [
        'color' => ['color'],
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public static function propertyDefinitions(FieldStorageDefinitionInterface $field_definition): array {
    $properties = [];
    $properties['color'] = DataDefinition::create('string')
      ->setLabel(new TranslatableMarkup('Color'));

    return $properties;
  }

  /**
   * {@inheritdoc}
   */
  public static function generateSampleValue(FieldDefinitionInterface $field_definition): array {
    return [
      'color' => 'blue-france-main-525',
    ];
  }

}
