<?php

declare(strict_types=1);

namespace Drupal\dsfr4drupal_colors\Plugin\Field\FieldWidget;

use Drupal\Core\Field\Attribute\FieldWidget;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;

/**
 * Plugin implementation of the dsfr4drupal_color field box widget.
 */
#[FieldWidget(
  id: 'dsfr4drupal_color_field_widget_box',
  label: new TranslatableMarkup('DSFR for Drupal - Color boxes'),
  field_types: ['dsfr4drupal_color_field_type'],
)]
class ColorFieldWidgetBox extends ColorFieldWidgetBase {

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings(): array {
    return [
        'squares_size' => 'medium',
        'has_search' => FALSE,
      ] + parent::defaultSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state): array {
    $elements = [];

    $elements['squares_size'] = [
      '#type' => 'radios',
      '#title' => $this->t('Squares size'),
      '#options' => $this->getSquareSizeOptions(),
      '#default_value' => $this->getSetting('squares_size'),
    ];

    $elements['has_search'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Display search input'),
      '#default_value' => $this->getSetting('has_search'),
    ];


    return $elements;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary(): array {
    $summary = [];

    $summary[] = $this->t('Squares size: @value', [
      '@value' => $this->getSquareSizeOptions()[$this->getSetting('squares_size')],
    ]);

    $summary[] = $this->t('Display search input: @value', [
      '@value' => $this->getSetting('has_search') ? $this->t('yes') : $this->t('no'),
    ]);

    return $summary;
  }

  /**
   * {@inheritdoc}
   */
  public function formElement(FieldItemListInterface $items, $delta, array $element, array &$form, FormStateInterface $form_state): array {
    $element = parent::formElement($items, $delta, $element, $form, $form_state);

    $element['color']['#type'] = 'dsfr4drupal_color_box';
    $element['color']['#allowed_colors'] = $this->getColorsOptions();
    $element['color']['#has_search'] = $this->getSetting('has_search');
    $element['color']['#squares_size'] = $this->getSetting('squares_size');

    return $element;
  }

  /**
   * Get available square size as options.
   *
   * @return array
   *   The square size options.
   */
  private function getSquareSizeOptions(): array {
    return [
      'small' => $this->t('Small'),
      'medium' => $this->t('Medium'),
      'large' => $this->t('Large'),
    ];
  }

}
