<?php

declare(strict_types=1);

namespace Drupal\dsfr4drupal_colors\Plugin\Field\FieldWidget;

use Drupal\Component\Utility\Html;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\WidgetBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\dsfr4drupal_colors\Helper\ColorsHelperInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Base class for dsfr4drupal_color widgets.
 */
abstract class ColorFieldWidgetBase extends WidgetBase {

  /**
   * The colors' helper.
   *
   * @var \Drupal\dsfr4drupal_colors\Helper\ColorsHelperInterface
   */
  protected ColorsHelperInterface $helper;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    $instance = parent::create(
      $container,
      $configuration,
      $plugin_id,
      $plugin_definition,
    );
    $instance->helper = $container->get('dsfr4drupal_colors.helper.colors');

    return $instance;
  }


  /**
   * {@inheritdoc}
   */
  public function formElement(FieldItemListInterface $items, $delta, array $element, array &$form, FormStateInterface $form_state): array {
    // Force identifier.
    $element['#id'] = $this->getHtmlIdentifier($this->fieldDefinition->getName(), $delta);

    $element['color'] = $element + [
      '#type' => 'textfield',
      '#required' => $element['#required'],
      '#default_value' => $items[$delta]->color ?? '',
    ];

    // Replace wrapper fieldset with container.
    $element['#type'] = 'container';

    if ($this->getFieldSetting('contrast_ratio_type')) {
      // Load contrast ratio libray.
      $element['#attached']['library'][] = 'dsfr4drupal_colors/contrast-ratio';

      // Pass all colors name with hexa.
      $element['#attached']['drupalSettings']['dsfr4drupal_colors']['colorsAvailable'] = array_flip($this->helper->getColorsAvailable());

      // Attach contrast ratio field settings.
      $value = $this->getFieldSetting('contrast_ratio_value');
      if ($this->getFieldSetting('contrast_ratio_type') === 'field') {
        // Estimate field identifier (only works with the first element).
        $value = $this->getHtmlIdentifier($value);
      }

      $element['#attached']['drupalSettings']['dsfr4drupal_colors']['contrastRatio'][$element['#id']] = [
        'type' => $this->getFieldSetting('contrast_ratio_type'),
        'value' => $value,
      ];
    }

    return $element;
  }

  /**
   * Get field HTML identifier.
   *
   * @param string $fieldName
   *   The field machine name.
   * @param int $delta
   *   The element delta.
   *
   * @return string
   *   The HTML identifier.
   */
  protected function getHtmlIdentifier(string $fieldName, int $delta = 0): string {
    return Html::getId('dsfr4drupal-color-' . $fieldName . '-' . $delta);
  }

  /**
   * Get selectable colors options.
   *
   * @return array
   *   The selectable colors options.
   */
  protected function getColorsOptions(): array {
    $allowedColors = $this->getFieldSetting('allowed_colors');
    $colors = $this->helper->getColorsOptions();

    if (!empty($allowedColors)) {
      foreach ($colors as $name => $label) {
        if (!in_array($name, $allowedColors)) {
          unset($colors[$name]);
        }
      }
    }

    return $colors;
  }

}
