/**
 * @file
 * Attaches behaviors for DSFR color field box widget.
 */

((Drupal, once) => {
  "use strict";

  const DATA_ATTRIBUTE_PREFIX = "data-color-widget-box";

  const CLASS_PREFIX = "dsfr4drupal-color-widget-box";
  const CLASS_ACTIVE = `${CLASS_PREFIX}-active`;
  const CLASS_HIDDEN  = `${CLASS_PREFIX}-hidden`;
  const CLASS_INPUT_SEARCH = `${CLASS_PREFIX}--input-search`;
  const CLASS_SQUARE = `${CLASS_PREFIX}--square`;
  const CLASS_SQUARE_TRANSPARENT = `${CLASS_SQUARE}__transparent`;
  const CLASS_WRAPPER = `${CLASS_PREFIX}-wrapper`;
  const SELECTOR_FORM_ELEMENT = `input.${CLASS_PREFIX}--element`;

  /**
   * Add a color picker widget.
   *
   * @param {Element} wrapper
   *   The wrapper element.
   * @param {Object} options
   *   Override default settings with custom options.
   */
  const addColorPicker = (wrapper, options) => {

    const DEFAULT_SETTINGS = {
      currentColor: "",
      hasSearch: false,
      squareElemType: "button",
      squareClass: "color-box",
      squaresSize: "medium",
      squareTransparentClass: "transparent-box",
      addTransparentSquare: true,
      clickCallback: (ignoredColor) => {
        return ignoredColor;
      },
      iterationCallback: null,
      fillString: "&nbsp;",
      fillStringX: "?",
      colors: ["#fff", "#000", "#ff0", "#f0f", "#f00", "#0ff", "#00f", "#0f0"],
    };

    if (!options) {
      options = [];
    }

    const settings = Object.assign(DEFAULT_SETTINGS, options);

    // Display a search input.
    if (settings.hasSearch) {
      const input = document.createElement("input");
      input.setAttribute("type", "search");
      input.classList.add("form-text", CLASS_INPUT_SEARCH);
      wrapper.append(input);

      input.addEventListener("input", (event) => {
        const value = input.value.toLowerCase().replace(' ', '-');

        wrapper.querySelectorAll(`.${CLASS_SQUARE}`).forEach((square) => {
          if (
            value.length === 0 ||
            square.getAttribute(`${DATA_ATTRIBUTE_PREFIX}-value`).toLowerCase().includes(value)
          ) {
            square.classList.remove(CLASS_HIDDEN);
          } else {
            square.classList.add(CLASS_HIDDEN);
          }
        });
      });
    }

    /**
     * Add a square element for given color.
     *
     * @param {string} colorName
     *   The given color name.
     * @param {string} colorLabel
     *   The given color label.
     *
     *   @return {Element}
     *     The added square element.
     */
    this.addSquareElement = (colorName, colorLabel) => {
      const element = document.createElement(settings.squareElemType);
      element.classList.add(
        settings.squareClass,
        `${settings.squareClass}--${settings.squaresSize}`
      );
      element.setAttribute(`${DATA_ATTRIBUTE_PREFIX}-value`, colorName);
      element.setAttribute("title", colorLabel);
      element.style.backgroundColor = `var(--${colorName})`;

      if (settings.currentColor === colorName) {
        element.classList.add(CLASS_ACTIVE);
      }

      if (settings.clickCallback) {
        element.addEventListener("click", (event) => {
          event.preventDefault();


          const elementActive = element.closest("div").querySelector(`${settings.squareElemType}.${CLASS_ACTIVE}`)
          if (elementActive) {
            elementActive.classList.remove(CLASS_ACTIVE);
          }

          element.classList.add(CLASS_ACTIVE);
          settings.clickCallback(element.getAttribute(`${DATA_ATTRIBUTE_PREFIX}-value`));
        });
      }
      wrapper.append(element);

      if (settings.iterationCallback) {
        settings.iterationCallback(this, element, colorName, colorLabel);
      }

      return element;
    };

    Object.entries(settings.colors).forEach(([colorName, colorLabel]) => {
      this.addSquareElement(colorName, colorLabel);
    })

    if (settings.addTransparentSquare) {
      this.addSquareElement(
        "",
        Drupal.t("none", {}, {context: "dsfr4drupal_colors none label"})
      )
        .classList.add(settings.squareTransparentClass);
    }
  };

  Drupal.behaviors.dsfr4drupalColorFieldWidgetBox = {
    attach: (context, settings) => {
      once("dsfr4drupal-color-field-widget-box", SELECTOR_FORM_ELEMENT, context)
        .forEach((input) => {
          const settingsColors = settings.dsfr4drupal_colors;
          const settingsBox = settingsColors.widgetBox[input.id] || {};

          // Hide input.
          input.classList.add("visually-hidden");

          // Add color picker wrapper.
          const wrapper = document.createElement("div");
          wrapper.classList.add(CLASS_WRAPPER);
          input.after(wrapper);

          // Use dark theme colors instead of default colors (= light theme).
          if (settingsColors.scheme === "dark") {
            wrapper.classList.add(settingsColors.classThemeDark);
          }
          else {
            wrapper.classList.add(settingsColors.classThemeLight);
          }

          addColorPicker(wrapper, Object.assign(settingsBox, {
            currentColor: input.value,
            squareClass: CLASS_SQUARE,
            squareTransparentClass: CLASS_SQUARE_TRANSPARENT,
            addTransparentSquare: !settingsBox.required,
            clickCallback: (color) => {
              input.value = color;

              // The "change" event is not automatically triggered when the field is hidden.
              // Trigger it manually.
              input.dispatchEvent(new Event("change"));
            },
          }));
        });
    },
  };

})(Drupal, once);
