<?php

declare(strict_types=1);

namespace Drupal\dsfr4drupal_colors\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\dsfr4drupal_colors\Helper\ColorsHelperInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

class ColorsController extends ControllerBase {

  /**
   * The colors' helper.
   *
   * @var \Drupal\dsfr4drupal_colors\Helper\ColorsHelperInterface
   */
  protected ColorsHelperInterface $helper;

  public static function create(ContainerInterface $container) {
    $instance = parent::create($container);
    $instance->helper = $container->get('dsfr4drupal_colors.helper.colors');

    return $instance;
  }

  /**
   * Display all available DSFR colors.
   *
   * @return array
   *  A renderable array.
   */
  public function collection(): array {
    $colors = $this->helper->getColorsAvailable();

    $items = [];
    foreach ($colors as &$color) {
      $items[] = [
        'swatch' => [
          '#type' => 'component',
          '#component' => 'dsfr4drupal_colors:color-swatch',
          '#props' => [
            'color' => $this->helper->getColorCssVariable($color),
            'width' => '30px',
            'height' => '30px',
          ],
        ],
        'label' => [
          '#markup' => $color,
        ]
      ];
    }

    $list = [
      '#theme' => 'item_list',
      '#items' => $items,
      '#attributes' => ['class' => ['dsfr4drupal-colors-collection--list']],
    ];

    return [
      'header' => [
        '#type' => 'html_tag',
        '#tag' => 'p',
        '#value' => $this->t(
          'DSFR provides <strong>@count</strong> colors. You can find the list of colors in light and dark modes below.',
          ['@count' => count($colors)]
        ),
      ],
      'light' => [
        '#type' => 'details',
        '#title' => $this->t('Colors in light mode'),
        '#open' => TRUE,
        'list' => $list + [
          '#theme_wrappers' => [
            'container' => [
              '#attributes' => ['class' => ColorsHelperInterface::CLASS_THEME_LIGHT],
            ],
          ],
        ],
      ],
      'dark' => [
        '#type' => 'details',
        '#title' => $this->t('Colors in dark mode'),
        '#open' => FALSE,
        'list' => $list + [
          '#theme_wrappers' => [
            'container' => [
              '#attributes' => ['class' => ColorsHelperInterface::CLASS_THEME_DARK],
            ],
          ],
        ],
      ],
      '#attached' => [
        'library' => ['dsfr4drupal_colors/collection'],
      ]
    ];
  }



}
