<?php

declare(strict_types=1);

namespace Drupal\dsfr4drupal_colors\Element;

use Drupal\Core\Render\Attribute\FormElement;
use Drupal\Core\Render\Element\Textfield;
use Drupal\dsfr4drupal_colors\Helper\ColorsHelperInterface;

/**
 * Provides a color form element.
 *
 * Usage example:
 *
 * @code
 * $form['color'] = [
 *   '#type' => 'dsfr4drupal_color_box',
 *   '#title' => $this->t('Color'),
 *   '#default_value' => 'blue-france-main-525',
 *   '#allowed_colors' => ['blue-france-main-525', 'red-marianne-main-472'],
 *   '#squares_size' => 'large',
 *   '#has_search' => TRUE,
 * ];
 * @endcode
 */
#[FormElement('dsfr4drupal_color_box')]
class ColorBoxElement extends Textfield {

  /**
   * {@inheritdoc}
   */
  public function getInfo(): array {
    $class = static::class;
    $info = parent::getInfo();

    $info['#pre_render'][] = [$class, 'preRenderColorBox'];

    // Limit displayed colors.
    $info['#allowed_colors'] = [];

    // Display search input.
    $info['#has_search'] = FALSE;

    // Define squares size (managed values: "small", "medium" or "large").
    $info['#squares_size'] = 'medium';

    return $info;
  }

  /**
   * Prepares a #type 'dsfr4drupal_color_box' render element.
   *
   * @param array $element
   *   An associative array containing the properties of the element.
   *   Properties used: #name, #value, #attributes.
   *
   * @return array
   *   The $element with prepared variables ready for select.html.twig.
   */
  public static function preRenderColorBox(array $element): array {
    $settings = [];

    // If the element is required but has no default value and the element is
    // hidden like the color boxes widget does, prevent HTML5 Validation from
    // being invisible and blocking save with no apparent reason.
    $element['#attributes']['class'][] = 'dsfr4drupal-color-widget-box--element';

    // Set Drupal settings.
    $settings[$element['#id']] = [
      'hasSearch' => $element['#has_search'],
      'required' => $element['#required'],
      'squaresSize' => $element['#squares_size'],
    ];

    $colors = \Drupal::service('dsfr4drupal_colors.helper.colors')
      ->getColorsOptions();

    if (!empty($element['#allowed_colors'])) {
      foreach ($colors as $name => $label) {
        if (!in_array($name, $element['#allowed_colors'])) {
          unset($colors[$name]);
        }
      }
    }

    foreach ($colors as $name => $label) {
      $settings[$element['#id']]['colors'][$name] = $label;
    }

    // Define which color palette is used.
    $element['#attached']['drupalSettings']['dsfr4drupal_colors']['scheme'] =
      \Drupal::config('dsfr4drupal_colors.settings')->get('scheme');

    $element['#attached']['library'][] = 'dsfr4drupal_colors/color-field-widget-box';
    $element['#attached']['drupalSettings']['dsfr4drupal_colors']['classThemeDark'] = ColorsHelperInterface::CLASS_THEME_DARK;;
    $element['#attached']['drupalSettings']['dsfr4drupal_colors']['classThemeLight'] = ColorsHelperInterface::CLASS_THEME_LIGHT;;
    $element['#attached']['drupalSettings']['dsfr4drupal_colors']['widgetBox'] = $settings;

    return $element;
  }

}
