<?php

declare(strict_types=1);

namespace Drupal\dsfr4drupal_colors\Plugin\Field\FieldFormatter;

use Drupal\Core\Field\Attribute\FieldFormatter;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;

/**
 * Plugin implementation of the dsfr4drupal_color field swatch formatter.
 */
#[FieldFormatter(
  id: 'dsfr4drupal_color_field_formatter_swatch',
  label: new TranslatableMarkup('Color swatch'),
  field_types: [
    'dsfr4drupal_color_field_type',
  ],
)]
class ColorFieldFormatterSwatch extends ColorFieldFormatterBase {

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings(): array {
    return [
        'shape' => 'square',
        'width' => 50,
        'height' => 50,
      ] + parent::defaultSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state): array {
    $elements = [];

    $elements['shape'] = [
      '#type' => 'select',
      '#title' => $this->t('Shape'),
      '#options' => $this->getShapesOptions(),
      '#default_value' => $this->getSetting('shape'),
    ];
    $elements['width'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Width'),
      '#default_value' => $this->getSetting('width'),
      '#min' => 1,
      '#description' => $this->t('Defaults to pixels (px) if a number is entered, otherwise, you can enter any unit (ie %, em, vw).'),
    ];
    $elements['height'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Height'),
      '#default_value' => $this->getSetting('height'),
      '#min' => 1,
      '#description' => $this->t('Defaults to pixels (px) if a number is entered, otherwise, you can enter any unit (ie %, em, vh).'),
    ];

    return $elements;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary(): array {
    $settings = $this->getSettings();
    $summary = [];

    $summary[] = $this->t('Shape: @shape', [
      '@shape' => $this->getShapesOptions()[$this->getSetting('shape')],
    ]);

    $summary[] = $this->t('Width: @width', [
      '@width' => $settings['width'],
    ]);

    $summary[] = $this->t('Height: @height', [
      '@height' => $settings['height'],
    ]);

    return $summary;
  }

  /**
   * {@inheritdoc}
   */
  public function viewElements(FieldItemListInterface $items, $langcode): array {
    $settings = $this->getSettings();

    $elements = [];

    foreach ($items as $delta => $item) {
      $elements[$delta] = [
        '#type' => 'component',
        '#component' => 'dsfr4drupal_colors:color-swatch',
        '#props' => [
          'color' => $this->viewValue($item),
          'shape' => $settings['shape'],
          'width' => is_numeric($settings['width']) ? "{$settings['width']}px" : $settings['width'],
          'height' => is_numeric($settings['height']) ? "{$settings['height']}px" : $settings['height'],
        ],
      ];
    }

    return $elements;
  }

  /**
   * Ge shapes options.
   *
   * @return \Drupal\Component\Render\MarkupInterface[]
   *   An array of shape ids/names.
   */
  protected function getShapesOptions(?string $shape = NULL): array {
    return [
      'circle' => $this->t('Circle'),
      'parallelogram' => $this->t('Parallelogram'),
      'square' => $this->t('Square'),
      'triangle' => $this->t('Triangle'),
    ];
  }

}
