/**
 * @file
 * Manage editing behavrios for Ckeditor5 DSFR picker plugins.
 */
/* eslint-disable max-nested-callbacks */

import { Plugin } from "ckeditor5/src/core";
import { Widget, toWidget } from "ckeditor5/src/widget";
import InsertIconCommand from "./command";
import { getPreviewContainer, previewIcon } from "./utils";

/**
 * @private
 */
export default class DsfrIconPickerEditing extends Plugin {

  /**
   * {@inheritdoc}
   */
  static get requires() {
    return [Widget];
  }

  /**
   * {@inheritdoc}
   */
  static get pluginName() {
    return "DsfrIconPickerEditing";
  }

  /**
   * {@inheritdoc}
   */
  init() {
    const options = this.editor.config.get("dsfrIconPicker");
    if (!options) {
      throw new Error(
        "Error on initializing \"DSFR icons\" plugin: dsfrIconPicker config is required.",
      );
    }

    this.options = options;

    this.modelName = "dsfrIcon";
    this.viewName = "dsfr-icon";

    this.attrs = {
      dataIcon: "data-icon",
      dataSize: "data-size",
    };
    this.converterAttributes = [
      "dataIcon",
      "dataSize",
    ];

    this._defineSchema();
    this._defineConverters();

    this.editor.commands.add(
      "insertIcon",
      new InsertIconCommand(this.editor),
    );
  }

  /**
   * Registers drupalEntity as a block element in the DOM.
   *
   * @private
   */
  _defineSchema() {
    const { schema } = this.editor.model;

    schema.register(this.modelName, {
      // Behaves like a self-contained inline object (e.g. an inline image)
      // allowed in places where $text is allowed (e.g. in paragraphs).
      // The inline widget can have the same attributes as text (for example linkHref, bold).
      inheritAllFrom: "$inlineObject",
      // Cannot be split or left by the caret.
      isLimit: true,
      // Restrict allowed attributes.
      allowAttributes: Object.keys(this.attrs),
    });
  }

  /**
   * Defines handling of DSFR icon element in the content lifecycle.
   *
   * @private
   */
  _defineConverters() {
    const { conversion } = this.editor;

    conversion.for("upcast").elementToElement({
      model: this.modelName,
      view: {
        name: this.viewName,
      },
    });
    conversion.for("dataDowncast").elementToElement({
      model: this.modelName,
      view: {
        name: this.viewName,
      },
    });
    conversion.for("dataDowncast").elementToElement({
      model: this.modelName,
      view: {
        name: this.viewName,
      },
    });
    conversion
      .for("editingDowncast")
      .elementToElement({
        model: this.modelName,
        view: (modelElement, { writer }) => {
          const container = writer.createContainerElement("figure", {
            class: "dsfr-icon",
          });
          writer.setCustomProperty(this.modelName, true, container);
          return toWidget(container, writer, { label: this.options.buttonLabel });
        }
      })
      .add((dispatcher) => {
        const converter = (event, data, conversionApi) => {
          const viewWriter = conversionApi.writer;
          const modelElement = data.item;
          const container = conversionApi.mapper.toViewElement(modelElement);

          let icon = getPreviewContainer(container.getChildren());

          // Use pre-existing icon preview container if one exists. If the
          // preview element doesn't exist, create a new element.
          if (icon) {
            // Stop processing if icon preview is unavailable or a preview is
            // already loading.
            if (icon.getAttribute("data-dsfr-icon-preview") !== "ready") {
              return;
            }

            // Preview was ready meaning that a new preview can be loaded.
            // "Change the attribute to loading to prepare for the loading of
            // the updated preview. Preview is kept intact so that it remains
            // interactable in the UI until the new preview has been rendered.
            viewWriter.setAttribute(
              "data-dsfr-icon-preview",
              "loading",
              icon,
            );
          }
          else {
            icon = viewWriter.createRawElement("div", {
              "data-dsfr-icon-preview": "loading",
            });
            viewWriter.insert(viewWriter.createPositionAt(container, 0), icon);
          }

          this._fetchPreview(modelElement).then((preview) => {
            if (!icon) {
              // Nothing to do if associated preview wrapped no longer exist.
              return;
            }
            // CKEditor 5 doesn't support async view conversion. Therefore, once
            // the promise is fulfilled, the editing view needs to be modified
            // manually.
            this.editor.editing.view.change((writer) => {
              const iconPreview = writer.createRawElement(
                "div",
                {
                  "data-dsfr-icon-preview": "ready",
                  class: "dsfr-icon-preview",
                },
                (domElement) => {
                  domElement.innerHTML = preview;
                },
              );
              // Insert the new preview before the previous preview element to
              // ensure that the location remains same even if it is wrapped
              // with another element.
              writer.insert(writer.createPositionBefore(icon), iconPreview);
              writer.remove(icon);
            });
          });
        };

        this.converterAttributes.forEach((attribute) => {
          dispatcher.on(`attribute:${attribute}:${this.modelName}`, converter);
        });

        return dispatcher;
      });

    Object.keys(this.attrs).forEach((modelKey) => {
      const attributeMapping = {
        model: {
          key: modelKey,
          name: this.modelName,
        },
        view: {
          name: this.viewName,
          key: this.attrs[modelKey],
        },
      };
      conversion.for("dataDowncast").attributeToAttribute(attributeMapping);
      conversion.for("upcast").attributeToAttribute(attributeMapping);
    });
  }

  /**
   * Fetches the preview.
   *
   * @param {HTMLElement} modelElement
   *   The model element.
   *
   * @private
   */
  async _fetchPreview(modelElement) {
    return previewIcon(modelElement);
  }

}
