/**
 * @file
 * Provides several functions used by Ckeditor5 DSFR picker plugins.
 */

import { isWidget } from "ckeditor5/src/widget";

/**
 * Create icon into Ckeditor content.
 *
 * @param {module:engine/model/writer~Writer} writer
 *   The CkEditor writer.
 * @param {object} attributes
 *   The icon attributes.
 * @param {string} element
 *   The CkEditor plugin element name.
 *
 * @return {module:engine/model/element~Element}
 *   The `dsfrIcon` element.
 */
export const createIcon = (writer, attributes, element) => {
  return writer.createElement(element, attributes);
}

/**
 * Preview DSFR icon.
 *
 * @param {Element} element
 *   The icon element.
 *
 * @return {string}
 *   The icon preview.
 */
export const previewIcon = (element) => {
  const classes = [
    element.getAttribute("dataIcon")
  ];

  const size = element.getAttribute("dataSize") ?? "md";
  classes.push("fr-icon--" + size);

  return "<span class=\"" + classes.join(" ") + "\"></span>";
}

/**
 * Gets selected DSFR Icon widget if only DSFR Icon is currently selected.
 *
 * @param {module:engine/model/selection~Selection} selection
 *   The current selection.
 * @return {module:engine/view/element~Element|null}
 *   The currently selected DSFR Icon widget or null.
 *
 * @private
 */
export function getClosestSelectedDsfrIconWidget(selection) {
  const viewElement = selection.getSelectedElement();
  if (viewElement && isDsfrIconWidget(viewElement)) {
    return viewElement;
  }

  // Perhaps nothing is selected.
  if (selection.getFirstPosition() === null) {
    return null;
  }

  let parent = selection.getFirstPosition().parent;

  while (parent) {
    if (parent.is("element") && isDsfrIconWidget(parent)) {
      return parent;
    }

    parent = parent.parent;
  }

  return null;
}

/**
 * Gets the preview container element from the icon element.
 *
 * @param {Iterable.<module:engine/view/element~Element>} children
 *   The child elements.
 * @return {null|module:engine/view/element~Element}
 *   The preview child element if available.
 */
export function getPreviewContainer(children) {
  // eslint-disable-next-line no-restricted-syntax
  for (const child of children) {
    if (child.hasAttribute("data-dsfr-icon-preview")) {
      return child;
    }

    if (child.childCount) {
      const recursive = getPreviewContainer(child.getChildren());
      // Return only if preview container was found within this element"s
      // children.
      if (recursive) {
        return recursive;
      }
    }
  }

  return null;
}

/**
 * Checks if view element is <dsfr-icon> element.
 *
 * @param {module:engine/view/element~Element} viewElement
 *   The view element.
 * @return {boolean}
 *   A boolean indicating if the element is a <dsfr-icon> element.
 */
export const isDsfrIconWidget = (viewElement) => {
  return (
    isWidget(viewElement) && !!viewElement.getCustomProperty("dsfrIcon")
  );
}

/**
 * Open a Drupal dialog.
 *
 * @param {object} options
 *   The dialog options.
 * @param {function} saveCallback
 *   The save callback method.
 * @param {object} existingValues
 *   The existing values.
 */
export const openDialog = (options, saveCallback, existingValues) => {
  const ckeditorAjaxDialog = Drupal.ajax(
    {
      dialog: {
        width: options.dialogSettings.width,
        height: options.dialogSettings.height,
        title: options.dialogSettings.title,
        class: options.class,
      },
      dialogType: "modal",
      selector: options.dialogSettings.selector,
      url: options.dialogUrl,
      progress: { type: "fullscreen" },
      submit: existingValues,
    }
  );
  ckeditorAjaxDialog.execute();

  // Store the save callback to be executed when this dialog is closed.
  Drupal.ckeditor5.saveCallback = saveCallback;
};
