/**
 * @file
 * Provides several functions used by Ckeditor5 DSFR picker plugins.
 */

import { isWidget } from "ckeditor5/src/widget";

/**
 * Create pictogram into Ckeditor content.
 *
 * @param {module:engine/model/writer~Writer} writer
 *   The CkEditor writer.
 * @param {object} attributes
 *   The pictogram attributes.
 * @param {string} element
 *   The CkEditor plugin element name.
 *
 * @return {module:engine/model/element~Element}
 *   The `dsfrPictogram` element.
 */
export const createPictogram = (writer, attributes, element) => {
  return writer.createElement(element, attributes);
}

/**
 * Preview DSFR pictogram.
 *
 * @param {Element} element
 *   The pictogram element.
 * @param {string} pictoBasePath
 *   The pictogram base path.
 *
 * @return {string}
 *   The pictogram preview.
 */
export const previewPictogram = (element, pictoBasePath) => {
  // @todo: Get width, height and viewBox dinamycally.
  const path = pictoBasePath + "/" + element.getAttribute("dataPictogram") + ".svg";
  const infos = parseSvg(path);

  let svgAttributes = "";
  Object.entries(infos).forEach(([attribute, value]) => {
    svgAttributes += `${attribute}="${value}"`;
  });

  return `<svg class="fr-artwork" aria-hidden="true" ${svgAttributes}>
    <use class="fr-artwork-decorative" href="${path}#artwork-decorative"></use>
    <use class="fr-artwork-minor" href="${path}#artwork-minor"></use>
    <use class="fr-artwork-major" href="${path}#artwork-major"></use>
  </svg>`;
}

/**
 * Gets the preview container element from the pictogram element.
 *
 * @param {Iterable.<module:engine/view/element~Element>} children
 *   The child elements.
 * @return {null|module:engine/view/element~Element}
 *   The preview child element if available.
 */
export function getPreviewContainer(children) {
  // eslint-disable-next-line no-restricted-syntax
  for (const child of children) {
    if (child.hasAttribute("data-dsfr-pictogram-preview")) {
      return child;
    }

    if (child.childCount) {
      const recursive = getPreviewContainer(child.getChildren());
      // Return only if preview container was found within this element"s
      // children.
      if (recursive) {
        return recursive;
      }
    }
  }

  return null;
}

/**
 * Get SVG file content from URL.
 *
 * @param {string} url
 *   The SVG url.
 *
 * @return {string}
 *   The SVG file content.
 */
export const getSvg = (url) => {
  const xmlHttp = new XMLHttpRequest();
  xmlHttp.open("GET", url, false);
  xmlHttp.send();
  return xmlHttp.responseText;
};

export const parseSvg = (url) => {
  const parse = {};
  const data = getSvg(url);

  if (data) {
    const matchesWidth = data.match(/<svg .*?width=["'](-?[\d\.]+px)["']/)
    if (matchesWidth && matchesWidth.length >= 2) {
      parse.width = matchesWidth[1];
    }

    const matchesHeight = data.match(/<svg .*?height=["'](-?[\d\.]+px)["']/)
    if (matchesHeight && matchesHeight.length >= 2) {
      parse.height = matchesHeight[1];
    }

    const matchesViewBox = data.match(/<svg .*?viewBox=["'](-?[\d\.]+[, ]+-?[\d\.]+[, ][\d\.]+[, ][\d\.]+)["']/)
    if (matchesViewBox && matchesViewBox.length >= 2) {
      parse.viewBox = matchesViewBox[1];
    }
  }

  return parse;
}

/**
 * Checks if view element is <dsfr-pictogram> element.
 *
 * @param {module:engine/view/element~Element} viewElement
 *   The view element.
 * @return {boolean}
 *   A boolean indicating if the element is a <dsfr-pictogram> element.
 */
export const isDsfrPictogramWidget = (viewElement) => {
  return (
    isWidget(viewElement) && !!viewElement.getCustomProperty("dsfrPictogram")
  );
}

/**
 * Open a Drupal dialog.
 *
 * @param {object} options
 *   The dialog options.
 * @param {function} saveCallback
 *   The save callback method.
 * @param {object} existingValues
 *   The existing values.
 */
export const openDialog = (options, saveCallback, existingValues) => {
  const ckeditorAjaxDialog = Drupal.ajax(
    {
      dialog: {
        width: options.dialogSettings.width,
        height: options.dialogSettings.height,
        title: options.dialogSettings.title,
        class: options.class,
      },
      dialogType: "modal",
      selector: options.dialogSettings.selector,
      url: options.dialogUrl,
      progress: { type: "fullscreen" },
      submit: existingValues,
    }
  );
  ckeditorAjaxDialog.execute();

  // Store the save callback to be executed when this dialog is closed.
  Drupal.ckeditor5.saveCallback = saveCallback;
};
