<?php

declare(strict_types=1);

namespace Drupal\dsfr4drupal_picker\Form;

use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\filter\FilterFormatInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Ckeditor dialog form base to insert DSRR icon and pictogram.
 */
abstract class DialogFormBase extends FormBase {


  /**
   * The default dialog width.
   *
   * @var string
   */
  const DEFAULT_DIALOG_WIDTH = 'auto';

  /**
   * The default dialog height.
   *
   * @var string
   */
  const DEFAULT_DIALOG_HEIGHT = 'auto';

  /**
   * The picker element.
   *
   * @var string
   */
  const PICKER_ELEMENT = '';

  /**
   * The form constructor.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The embedded content plugin manager.
   */
  public function __construct(
    protected EntityTypeManagerInterface $entityTypeManager
  ) {
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('entity_type.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(
    array $form,
    FormStateInterface $form_state,
    ?FilterFormatInterface $filter_format = NULL
  ): array {
    $payload = $this->getRequest()->getPayload();

    $element = $form_state->getValue('element') ?? $payload->get('element') ?? $form_state->getUserInput()['element'] ?? NULL;

    $form['element'] = [
      '#type' => 'dsfr4drupal_picker_' . static::PICKER_ELEMENT,
      '#default_value' => $element,
      '#keep_open' => TRUE,
      '#required' => TRUE,
    ];

    /** @var \Drupal\editor\EditorInterface|null $editor */
    $editor = $this->entityTypeManager->getStorage('editor')
      ->load($filter_format->id());
    if ($editor) {
      $pluginSettings = $editor->getSettings()['plugins']['dsfr4drupal_picker_' . static::PICKER_ELEMENT] ?? [];

      $form['element']['#allowed_groups'] = $pluginSettings['allowed_groups'] ?? [];
      $form['element']['#has_search'] = (bool) $pluginSettings['has_search'] ?? TRUE;
    }

    $form['actions'] = [
      '#type' => 'actions',
      'submit' => [
        '#type' => 'button',
        '#value' => $this->t('Insert'),
        '#ajax' => [
          'callback' => [$this, 'ajaxSubmitForm'],
          'disable-refocus' => TRUE,
        ],
      ],
    ];

    return $form;
  }

  /**
   * Ajax submit callback to insert or replace the html in ckeditor.
   *
   * @param array $form
   *   The form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return \Drupal\Core\Ajax\AjaxResponse
   *   Ajax response for injecting html in ckeditor.
   */
  public function ajaxSubmitForm(array &$form, FormStateInterface $form_state): AjaxResponse {
    return new AjaxResponse();
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    // Don't do anything.
  }

}
