<?php

declare(strict_types=1);

namespace Drupal\dsfr4drupal_picker\Plugin\CKEditor5Plugin;

use Drupal\ckeditor5\Plugin\CKEditor5PluginConfigurableInterface;
use Drupal\ckeditor5\Plugin\CKEditor5PluginConfigurableTrait;
use Drupal\ckeditor5\Plugin\CKEditor5PluginDefault;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Extension\ModuleExtensionList;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Plugin\PluginFormFactoryInterface;
use Drupal\Core\Plugin\PluginFormInterface;
use Drupal\Core\Plugin\PluginWithFormsInterface;
use Drupal\dsfr4drupal_picker\Form\DialogFormBase;
use Drupal\dsfr4drupal_picker\PickerInterface;
use Drupal\dsfr4drupal_picker\PickerManagerInterface;
use Drupal\editor\EditorInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * CKEditor 5 DSFR plugin base.
 *
 * @internal
 *   Plugin classes are internal.
 */
abstract class PluginBase extends CKEditor5PluginDefault implements CKEditor5PluginConfigurableInterface, PluginWithFormsInterface, ContainerFactoryPluginInterface {
  use CKEditor5PluginConfigurableTrait;

  /**
   * The picker element.
   *
   * @var string
   */
  const PICKER_ELEMENT = '';

  /**
   * The CKEditor plugin name.
   *
   * @var string
   */
  const PLUGIN_NAME = '';

  /**
   * The plugin form.
   *
   * @var \Drupal\Core\Plugin\PluginFormInterface
   */
  protected PluginFormInterface $form;

  /**
   * The DSFR for Drupal picker plugin instance.
   *
   * @var \Drupal\dsfr4drupal_picker\PickerInterface
   */
  protected PickerInterface $picker;

  /**
   * Constructs an Icon object.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin ID for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Plugin\PluginFormFactoryInterface $pluginFormFactory
   *    The plugin form factory.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory.
   * @param \Drupal\dsfr4drupal_picker\PickerManagerInterface $picker_manager
   *   The DSFR for Drupal picker manager.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    PluginFormFactoryInterface $pluginFormFactory,
    protected ConfigFactoryInterface $configFactory,
    protected ModuleExtensionList $extensionList,
    protected PickerManagerInterface $picker_manager
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);

    $this->form = $pluginFormFactory->createInstance($this, 'configure');
    $this->picker = $this->picker_manager->createInstance(static::PICKER_ELEMENT);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('plugin_form.factory'),
      $container->get('config.factory'),
      $container->get('extension.list.module'),
      $container->get('plugin.manager.dsfr4drupal_picker')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return [
      'allowed_groups' => [],
      'has_search' => TRUE,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
    return $this->form->buildConfigurationForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateConfigurationForm(array &$form, FormStateInterface $form_state) {
    $this->form->validateConfigurationForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state) {
    $this->form->submitConfigurationForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function getDynamicPluginConfig(array $static_plugin_config, EditorInterface $editor): array {
    $settings = $this->configFactory->get('dsfr4drupal_picker.settings');
    $iconsAvailable = $this->picker->getItemsAvailable();

    $icons = [];
    foreach ($iconsAvailable as $group => $items) {
      if (
        !empty($this->configuration['allowed_groups']) &&
        !in_array($group, $this->configuration['allowed_groups'])
      ) {
        continue;
      }

      // Translate group label.
      $groupTranslated = $this->picker->getGrouplabel($group)
        ->render();

      // Add a list of items to the widget by category.
      $icons[$groupTranslated] = $items;
    }

    // Sort items' groups by name.
    $this->picker->sortGroupsByKey($icons);

    return [
      static::PLUGIN_NAME => [
        'buttonLabel' => '',
        'dialogSettings' => [
          'width' => DialogFormBase::DEFAULT_DIALOG_WIDTH,
          'height' => DialogFormBase::DEFAULT_DIALOG_HEIGHT,
          'title' => '',
        ],
        'dialogUrl' => '',
        'has_search' => $this->configuration['has_search'],
        'items' => $icons,
        'theme' => $settings->get('theme'),
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function hasFormClass($operation) {
    return TRUE;
  }

}
