<?php

declare(strict_types=1);

namespace Drupal\dsfr4drupal_picker\Plugin\Dsfr4DrupalPicker;

use Drupal\Core\Cache\UseCacheBackendTrait;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Plugin\PluginBase;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\dsfr4drupal_picker\PickerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a picker plugin base class.
 *
 * @package Drupal\dsfr4drupal_picker
 */
abstract class PickerBase extends PluginBase implements PickerInterface, ContainerFactoryPluginInterface {

  use UseCacheBackendTrait;
  use StringTranslationTrait;

  /**
   * Creates a Picker plugin instance.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin ID for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\File\FileSystemInterface $fileSystem
   *     The file system service.
   * @param \Drupal\Core\Extension\ModuleHandler $moduleHandler
   *    The module handler service.
   */
  final public function __construct(
    array $configuration,
    string $plugin_id,
    mixed $plugin_definition,
    protected FileSystemInterface $fileSystem,
    protected ModuleHandlerInterface $moduleHandler
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(
    ContainerInterface $container,
    array $configuration,
    $plugin_id,
    $plugin_definition
  ): static {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('file_system'),
      $container->get('module_handler'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getGroupLabel(string $group): TranslatableMarkup {
    return $this->t(ucfirst($group), [], ['context' => 'DSFR Group label']);
  }

  /**
   * {@inheritdoc}
   */
  public function getGroups(): array {
    $items = $this->getItemsAvailable();
    return array_keys($items);
  }

  /**
   * {@inheritdoc}
   */
  public function sortGroups(array &$groups): void {
    uasort($groups, function($stringA, $stringB) {
      return strcmp(
        iconv('UTF-8', 'ASCII//TRANSLIT', $stringA),
        iconv('UTF-8', 'ASCII//TRANSLIT', $stringB)
      );
    });
  }

  /**
   * {@inheritdoc}
   */
  public function sortGroupsByKey(array &$groups): void {
    uksort($groups, function($stringA, $stringB) {
      return strcmp(
        iconv('UTF-8', 'ASCII//TRANSLIT', $stringA),
        iconv('UTF-8', 'ASCII//TRANSLIT', $stringB)
      );
    });
  }

  /**
   * Sort items.
   *
   * @param array $items
   *   The items to sort.
   */
  protected function sortItems(array &$items): void {
    // Sort items' groups by name.
    ksort($items);

    foreach (array_keys($items) as $group) {
      // Sort items by name.
      asort($items[$group]);
    }
  }

}
