<?php

declare(strict_types=1);

namespace Drupal\dsfr4drupal_picker\Plugin\Dsfr4DrupalPicker;

use Drupal\dsfr4drupal_picker\Attribute\Dsfr4DrupalPicker;

/**
 * Provides a pictogram picker plugin.
 *
 * @package Drupal\dsfr4drupal_picker
 */
#[Dsfr4DrupalPicker("pictogram")]
class PictogramPicker extends PickerBase implements PictogramPickerInterface {

  /**
   * Get icon files in DSFR library.
   *
   * @return string[]
   *   The icon files path.
   */
  private function getFiles(): array {
    $files = $this->fileSystem->scanDirectory($this->getItemsPath(), '#(.+)\.svg$#', ['min_depth' => 1]);

    if (empty($files)) {
      throw new \LogicException(
        'Any icon files found. Make sure you have installed the DSFR library correctly.'
      );
    }

    return $files;
  }

  /**
   * {@inheritdoc}
   */
  public function getItemPath(string $item): string {
    // Value "group/pictogram" expected.

    $path = $this->getItemsPath() . '/' . $item . '.svg';

    // Allow to alter pictograms.
    $this->moduleHandler->alter(
      'dsfr4drupal_picker_pictogram_path',
      $item,
      $path
    );

    return $path;
  }

  /**
   * {@inheritdoc}
   */
  public function getItemsAvailable(): array {
    $cid = 'dsfr4drupal_picker:pictograms';
    $cache = $this->cacheGet($cid);
    if ($cache) {
      return $cache->data;
    }

    $pictograms = $this->getItemsOriginal();

    // Allow to add custom pictograms.
    $pictogramsExtended = $this->moduleHandler->invokeAll('dsfr4drupal_picker_pictograms');
    $pictograms = array_merge_recursive($pictograms, $pictogramsExtended);

    // Sort items.
    $this->sortItems($pictograms);

    // Allow to alter pictograms.
    $this->moduleHandler->alter('dsfr4drupal_picker_pictograms', $pictograms);

    $this->cacheSet($cid, $pictograms);

    return $pictograms;
  }

  /**
   * {@inheritdoc}
   */
  public function getItemsOriginal(): array {
    $cid = 'dsfr4drupal_picker:pictograms:original';
    $cache = $this->cacheGet($cid);
    if ($cache) {
      return $cache->data;
    }

    $files = $this->getFiles();

    $pictograms = [];
    foreach ($files as &$file) {
      $group = basename(dirname($file->uri));

      if (!isset($pictograms[$group])) {
        $pictograms[$group] = [];
      }

      $pictograms[$group][] = $group . '/' . $file->name;
    }

    // Sort items.
    $this->sortItems($pictograms);

    $this->cacheSet($cid, $pictograms);

    return $pictograms;
  }

  /**
   * {@inheritdoc}
   */
  public function getItemsPath(): string {
    return self::LIBRARY_PATH . 'artwork/pictograms';
  }

}
