<?php

declare(strict_types=1);

namespace Drupal\dsfr4drupal_picker\Plugin\Field\FieldFormatter;

use Drupal\Component\Utility\Html;
use Drupal\Core\Field\Attribute\FieldFormatter;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\FormatterBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;

/**
 * Plugin implementation of the 'dsfr4drupal_picker_icon' formatter.
 */
#[FieldFormatter(
  id: 'dsfr4drupal_picker_icon',
  label: new TranslatableMarkup('DSFR for Drupal - Icon Picker'),
  field_types: [
    'dsfr4drupal_picker_icon',
  ],
)]
class IconPickerFormatter extends FormatterBase {

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings() {
    return [
        'size' => 'md',
      ] + parent::defaultSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state) {
    $form = parent::settingsForm($form, $form_state);

    $form['size'] = [
      '#type' => 'select',
      '#title' => $this->t('Size'),
      '#required' => TRUE,
      '#default_value' => $this->getSetting('size'),
      '#options' => $this->getSizeOptions(),
      '#attributes' => [
        'class' => ['container-inline'],
      ],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary() {
    $summary = parent::settingsSummary();

    $summary[] = $this->t(
      'Size: @size',
      ['@size' => $this->getSizeOptions()[$this->getSetting('size')]]
    );

    return $summary;
  }

  /**
   * {@inheritdoc}
   */
  public function viewElements(FieldItemListInterface $items, $langcode): array {
    $elements = [];

    foreach ($items as $delta => $item) {
      $elements[$delta] = [
        '#type' => 'component',
        '#component' => 'dsfr4drupal_picker:icon',
        '#props' => [
          'icon' => Html::escape($item->value),
          'size' => $this->getSetting('size'),
        ],
      ];
    }

    return $elements;
  }

  /**
   * Get size options.
   *
   * @return array
   *   The size options.
   */
  private function getSizeOptions(): array {
    return [
      'xs' => $this->t('Extra Small'),
      'sm' => $this->t('Small'),
      'md' => $this->t('Medium'),
      'lg' => $this->t('Large'),
    ];
  }

}
