<?php

declare(strict_types=1);

namespace Drupal\dsfr4drupal_picker\Plugin\Field\FieldType;

use Drupal\Core\Field\FieldItemBase;
use Drupal\Core\Field\FieldStorageDefinitionInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\TypedData\DataDefinition;

/**
 * Plugin base of a DSFR icon picker field type.
 */
abstract class PickerBase extends FieldItemBase {

  /**
   * The picker element.
   *
   * @var string
   */
  const PICKER_ELEMENT = '';

  /**
   * {@inheritdoc}
   */
  public static function propertyDefinitions(FieldStorageDefinitionInterface $field_definition): array {
    // This is called very early by the user entity roles field. Prevent
    // early t() calls by using the TranslatableMarkup.
    return [
      'value' => DataDefinition::create('string')
        ->setLabel(new TranslatableMarkup('Value'))
        ->setRequired(TRUE),
    ];
  }

  /**
   * {@inheritdoc}
   */
  public static function schema(FieldStorageDefinitionInterface $field_definition): array {
    return [
      'columns' => [
        'value' => [
          'type' => 'varchar',
          'length' => 100,
        ],
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public static function defaultFieldSettings(): array {
    return parent::defaultFieldSettings() + [
        'allowed_groups' => [],
      ];
  }

  /**
   * {@inheritdoc}
   */
  public function fieldSettingsForm(array $form, FormStateInterface $form_state): array {
    return [
      'allowed_groups' => [
        '#type' => 'checkboxes',
        '#title' => $this->t('Allowed groups'),
        '#description' => $this->t('Select the groups that can be picked. If no groups are checked, all groups will be available.'),
        '#options' => $this->getGroupsOptions(),
        '#default_value' => $this->getSetting('allowed_groups') ?: [],
        '#element_validate' => [[static::class, 'validateAllowedGroups']],
      ],
    ];
  }

  /**
   * Render API callback: Processes the allowed groups value.
   *
   * Ensure the element's value is an indexed array of selected group IDs.
   * This function is assigned as an #element_validate callback.
   *
   * @see static::fieldSettingsForm()
   */
  public static function validateAllowedGroups(array &$element, FormStateInterface $form_state) {
    $value = array_values(array_filter($form_state->getValue($element['#parents'])));
    $form_state->setValueForElement($element, $value);
  }

  /**
   * Get groups options.
   *
   * @return array
   *   The groups options.
   */
  protected function getGroupsOptions(): array{
    /** @var \Drupal\dsfr4drupal_picker\PickerInterface $picker */
    $picker = \Drupal::service('plugin.manager.dsfr4drupal_picker')->createInstance(static::PICKER_ELEMENT);

    $options = [];
    foreach ($picker->getGroups() as &$group) {
      $options[$group] = $picker->getGroupLabel($group)->render();
    }

    // Sort items' groups by name.
    $picker->sortGroups($options);

    return $options;
  }

  /**
   * {@inheritdoc}
   */
  public function isEmpty(): bool {
    $value = $this->get('value')->getValue();
    return $value === NULL || $value === '';
  }

}
