<?php

namespace Drupal\dsfr4drupal_picker\TwigExtension;

use Drupal\Core\Template\Attribute;
use Drupal\dsfr4drupal_picker\PickerManagerInterface;
use Drupal\dsfr4drupal_picker\Plugin\Dsfr4DrupalPicker\PictogramPickerInterface;
use Twig\Extension\AbstractExtension;
use Twig\TwigFilter;
use Twig\TwigFunction;

/**
 * Provides Twig function and filter.
 */
class PictogramExtension extends AbstractExtension {

  /**
   * The pictogram picker plugin instance.
   *
   * @var \Drupal\dsfr4drupal_picker\Plugin\Dsfr4DrupalPicker\PictogramPickerInterface
   */
  protected PictogramPickerInterface $picker;

  /**
   * Constructs an Extension object.
   *
   * @param \Drupal\dsfr4drupal_picker\PickerManagerInterface $picker_manager
   *    The DSFR for Drupal picker manager.
   *
   * @throws \Drupal\Component\Plugin\Exception\PluginException
   */
  public function __construct(PickerManagerInterface $picker_manager) {
    $this->picker = $picker_manager->createInstance('pictogram');
  }

  /**
   * {@inheritdoc}
   */
  public function getFilters(): array {
    return [
      new TwigFilter(
        'dsfr_pictogram_url',
        $this->getPictogramUrl(...),
      ),
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getFunctions(): array {
    return [
      new TwigFunction(
        'svg_attributes',
        $this->getSvgAttributes(...),
      ),
    ];
  }

  /**
   * Generate DSFR pictogram URL.
   *
   * @param string $pictogram
   *   The pictogram to load (value "group/pictogram" expected).
   *
   * @return string|null
   *   The pictogram's path or NULL if pictogram doesn't exist.
   */
  public function getPictogramUrl(string $pictogram): ?string {
    $path = $this->picker->getItemPath(trim($pictogram, '/'));

    return file_exists($path) ? base_path() . $path : NULL;
  }

  /**
   * Get SVG attributes ("width", "height", "viewBox").
   *
   * @param string $filepath
   *   The SVG file path.
   *
   * @return \Drupal\Core\Template\Attribute
   *   The found attributes.
   */
  public function getSvgAttributes(string $filepath): Attribute {
    $filepath = ltrim($filepath, '/');

    // Parse file content as XML if it's possible.
    if (function_exists('simplexml_load_file')) {
      $xml = simplexml_load_file($filepath);

      return new Attribute([
        'width' => (string) ($xml['width'] ? str_replace('px', '', $xml['width']) : ''),
        'height' => (string) ($xml['height'] ? str_replace('px', '', $xml['height']) : ''),
        'viewBox' => (string) ($xml['viewBox'] ?? ''),
      ]);
    }

    // Otherwise, parse file content with regex.
    $data = file_get_contents($filepath);
    if ($data) {
      preg_match('#<svg .*?width=["\'](-?[\d\.]+)px["\']#', $data, $matchWidth);
      preg_match('#<svg .*?height=["\'](-?[\d\.]+)px["\']#', $data, $matchHeight);
      preg_match('#<svg .*?viewBox=["\'](-?[\d\.]+[, ]+-?[\d\.]+[, ][\d\.]+[, ][\d\.]+)["\']#', $data, $matchViewBox);
    }

    return new Attribute([
       'width' => $matchWidth[1] ?? '',
       'height' => $matchHeight[1] ?? '',
       'viewBox' => $matchViewBox[1] ?? '',
     ]);
  }

}
