/**
 *
 * @file
 * Provides several commands used by Ckeditor5 DSFR picker plugins.
 */

import { Command } from "ckeditor5/src/core";

/**
 * @private
 */
export default class DsfrIconPickerSizeCommand extends Command {

  /**
   * Executes the command. Applies the size `value` to the selected DSFR icon.
   *
   * @param {object} options
   *   Options for the executed command.
   * @param {string} options.value
   *   The value to apply.
   */
  execute( options) {
    const editor = this.editor;
    const model = editor.model;
    const document = model.document;

    // Update command value.
    this.value = options.value;

    model.change( writer => {
      this._setSizeOnSelection(document.selection.getSelectedElement(), writer, options.value);
    });
  }

  /**
   * {@inheritdoc}
   */
  refresh() {
    const element = this.editor.model.document.selection.getSelectedElement();
    // As first check whether to enable or disable the command as the value will always be false if the command cannot be enabled.
    this.isEnabled = Boolean(element) && this._canBeSized(element);
    this.value = "md";

    if (this.isEnabled && element.hasAttribute("dataSize")) {
      this.value = element.getAttribute("dataSize");
    }
  }

  /**
   * Checks whether a block can have alignment set.
   *
   * @param {module:engine/model/element~Element} element
   *   The element to be checked.
   *
   * @return {boolean}
   *   Element can be sized or not.
   *
   * @private
   */
  _canBeSized(element) {
    return this.editor.model.schema.checkAttribute( element, "dataSize");
  }

  /**
   * Sets the size attribute on icon element.
   *
   * @param {module:engine/model/element~Element} element
   *   The current selected element.
   * @param {module:engine/model/writer~Writer} writer
   *   The CkEditor writer.
   * @param {string} size
   *   The new icon size.
   *
   * @private
   */
  _setSizeOnSelection(element, writer, size) {
    writer.setAttribute("dataSize", size, element);
  }

}
