<?php

declare(strict_types=1);

namespace Drupal\dsfr4drupal_picker\Element;

use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Render\Element\Select;

/**
 * Provides an icon picker base form element.
 */
abstract class PickerBase extends Select {

  /**
   * The HTML data attribute prefix.
   *
   * @var string
   */
  const DATA_ATTRIBUTE_PREFIX = 'data-dsfr4drupal-picker-';

  /**
   * The picker element.
   *
   * @var string
   */
  const PICKER_ELEMENT = '';

  /**
   * {@inheritdoc}
   */
  public function getInfo(): array {
    $class = static::class;
    $info = parent::getInfo();

    $info['#process'] = array_merge([[$class, 'processPicker']], $info['#process']);
    $info['#pre_render'][] = [$class, 'preRenderPicker'];

    // Limit displayed icon groups.
    $info['#allowed_groups'] = [];

    // Add a search bar on widget.
    $info['#has_search'] = TRUE;

    // Keep widget open instead of display dropdown.
    $info['#keep_open'] = FALSE;

    return $info;
  }

  /**
   * Prepares a #type 'dsfr4drupal_picker' render element.
   *
   * @param array $element
   *   An associative array containing the properties of the element.
   *
   * @return array
   *   The $element with prepared variables ready for select.html.twig.
   */
  public static function preRenderPicker(array $element): array {
    $settings = \Drupal::config('dsfr4drupal_picker.settings');

    $element['#attributes']['class'][] = 'dsfr4drupal-picker-' . static::PICKER_ELEMENT . '-element';

    // Display empty icon when field is optional.
    $element['#attributes'][static::DATA_ATTRIBUTE_PREFIX . 'empty-icon'] = $element['#required'] ? '': 'true';

    // Display field with search bar.
    $element['#attributes'][static::DATA_ATTRIBUTE_PREFIX . 'has-search'] = $element['#has_search'] ? 'true' : '';

    // Keep widget open instead of display dropdown.
    $element['#attributes'][static::DATA_ATTRIBUTE_PREFIX . 'keep-open'] = $element['#keep_open'] ? 'true' : '';

    // Attach icon picker library.
    $element['#attached']['library'][] = 'dsfr4drupal_picker/form-element.' . static::PICKER_ELEMENT;

    $drupalSettings = &$element['#attached']['drupalSettings']['dsfr4drupal_picker'];
    $drupalSettings['theme'] = $settings->get('theme');
    $drupalSettings['data_attribute_prefix'] = static::DATA_ATTRIBUTE_PREFIX;

    return $element;
  }

  /**
   * Processes a #type 'dsfr4drupal_picker' render element.
   *
   * @param array $element
   *   The form element to process.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   * @param array $complete_form
   *   The complete form structure.
   *
   * @return array
   *   The processed element.
   *
   * @see _form_validate()
   */
  public static function processPicker(
    array &$element,
    FormStateInterface $form_state,
    array &$complete_form
  ): array {
    $drupalSettings = &$element['#attached']['drupalSettings']['dsfr4drupal_picker'];
    $drupalSettings['source_' . static::PICKER_ELEMENT] = [];

    // Define options.
    $element['#options'] = [];

    try {
      /** @var \Drupal\dsfr4drupal_picker\PickerInterface $picker */
      $picker = \Drupal::service('plugin.manager.dsfr4drupal_picker')->createInstance(static::PICKER_ELEMENT);
      $itemsAvailable = $picker->getItemsAvailable();

      foreach ($itemsAvailable as $group => $items) {
        if (
          !empty($element['#allowed_groups']) &&
          !in_array($group, $element['#allowed_groups'])
        ) {
          continue;
        }

        // Translate group label.
        $groupTranslated = $picker
          ->getGrouplabel($group)
          ->render();

        // Cannot use "optgroup" to prevent not allowed value during form validation
        // Flatten options process only works for "select" type.
        $element['#options'] += array_combine($items, $items);

        // Add a list of items to the widget by category.
        $drupalSettings['source_' . static::PICKER_ELEMENT][$groupTranslated] = $items;
      }

      // Sort items' groups by name.
      $picker->sortGroupsByKey($drupalSettings['source_' . static::PICKER_ELEMENT]);
    }
    catch (\LogicException $e) {
      \Drupal::service('logger.factory')->get('dsfr4drupal_picker')
        ->error($e->getMessage());
    }

    // Define empty value to empty string instead of "_none".
    $element['#empty_value'] = '';

    // Define default value if field is required.
    if ($element['#required'] && empty($element['#default_value'])) {
      $element['#default_value'] = key($element['#options']);
    }

    return $element;
  }

}
