<?php

declare(strict_types=1);

namespace Drupal\dsfr4drupal_picker\Plugin\Dsfr4DrupalPicker;

use Drupal\dsfr4drupal_picker\Attribute\Dsfr4DrupalPicker;

/**
 * Provides an icon picker plugin.
 *
 * @package Drupal\dsfr4drupal_picker
 */
#[Dsfr4DrupalPicker("icon")]
class IconPicker extends PickerBase implements IconPickerInterface {

  /**
   * Get icons CSS files.
   *
   * @return string[]
   *   The CSS files path.
   */
  private function getCssFiles(): array {
    $filesScan = $this->fileSystem->scanDirectory($this->getCssFilesPath(), '#(.+)\.main\.min\.css$#', ['min_depth' => 1]);

    if (!isset($filesScan)) {
      throw new \LogicException(
        'Any style sheet files found. Make sure you have configured your font style sheet path correctly.'
      );
    }

    $files = [];
    foreach ($filesScan as $fileScan) {
      $files[] = $fileScan->uri;
    }

    return $files;
  }

  /**
   * Get icons CSS files path.
   *
   * @return string
   *   The icons CSS files path.
   */
  public function getCssFilesPath(): string {
    return self::LIBRARY_PATH . 'utility/icons/';
  }

  /**
   * {@inheritdoc}
   */
  public function getItemPath(string $item): string {
    // There are no CSS file by icon.
    return $item;
  }

  /**
   * {@inheritdoc}
   */
  public function getItemsAvailable(): array {
    $cid = 'dsfr4drupal_picker:icons';
    $cache = $this->cacheGet($cid);
    if ($cache) {
      return $cache->data;
    }

    $icons = $this->getItemsOriginal();

    // Allow to add custom icons.
    $iconsExtended = $this->moduleHandler->invokeAll('dsfr4drupal_picker_icons');
    $icons = array_merge_recursive($icons, $iconsExtended);

    // Sort items.
    $this->sortItems($icons);

    // Allow to alter icons.
    $this->moduleHandler->alter('dsfr4drupal_picker_icons', $icons);

    $this->cacheSet($cid, $icons);

    return $icons;
  }

  /**
   * {@inheritdoc}
   */
  public function getItemsOriginal(): array {
    $cid = 'dsfr4drupal_picker:pictograms:original';
    $cache = $this->cacheGet($cid);
    if ($cache) {
      return $cache->data;
    }

    $files = $this->getCssFiles();

    $icons = [];
    foreach ($files as &$file) {
      $group = substr(basename(dirname($file)), strlen('icons-'));
      $groupIcons = $this->parseCssFile($file);

      $icons[$group] = array_merge($groupIcons, $groupIcons);
    }

    // Sort items.
    $this->sortItems($icons);

    $this->cacheSet($cid, $icons);

    return $icons;
  }

  /**
   * {@inheritdoc}
   */
  public function getItemsPath(): string {
    return self::LIBRARY_PATH . 'icons';
  }

  /**
   * Parse CSS file to find icons declarations.
   *
   * @param string $filepath
   *    The CSS filepath.
   *
   * @return array
   *   The icon class declarations.
   */
  private function parseCssFile(string $filepath): array {
    $content = file_get_contents($filepath);

    // Looking for "fr-icon" prefix ("fr-fi" is deprecated).
    if (preg_match_all(
      '#\.(fr-icon-[\w_-]+)#',
      $content,
      $matches
    )) {
      return array_unique($matches[1]);
    }

    return [];
  }

}
