<?php

declare(strict_types=1);

namespace Drupal\dsfr4drupal_picker\PluginForm;

use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\PluginFormBase;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\dsfr4drupal_picker\PickerInterface;
use Drupal\dsfr4drupal_picker\PickerManagerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a base form for configuring the CKEditor 5 DSFR Icon & Pictogram plugins.
 *
 * This form allows text format admins to configure:
 *  - the allowed groups.
 */
abstract class ConfigureFormBase extends PluginFormBase implements ContainerInjectionInterface {

  use StringTranslationTrait;

  /**
   * The picker element.
   *
   * @var string
   */
  const PICKER_ELEMENT = '';

  /**
   * The pictogram picker plugin instance.
   *
   * @var \Drupal\dsfr4drupal_picker\PickerInterface
   */
  protected PickerInterface $picker;

  /**
   * Constructs a ConfigureIconForm object.
   *
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $moduleHandler
   *   The module handler.
   * @param \Drupal\dsfr4drupal_picker\PickerManagerInterface $picker_manager
   *    The DSFR for Drupal picker manager.
   */
  public function __construct(
    protected ModuleHandlerInterface $moduleHandler,
    PickerManagerInterface $picker_manager
  ) {
    $this->picker = $picker_manager->createInstance(static::PICKER_ELEMENT);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('module_handler'),
      $container->get('plugin.manager.dsfr4drupal_picker')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $configuration = $this->plugin->getConfiguration();

    $groups = $this->picker->getGroups();
    $options = [];
    foreach ($groups as &$group) {
      $options[$group] = $this->picker->getGroupLabel($group)->render();
    }

    // Sorting groups after translating them.
    $this->picker->sortGroups($options);

    $form['allowed_groups'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Allowed groups'),
      '#description' => $this->t('Select the groups that can be picked. If no groups are checked, all groups will be available.'),
      '#options' => $options,
      '#default_value' => $configuration['allowed_groups'] ?? [],
    ];

    $form['has_search'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Display search input'),
      '#default_value' => $configuration['has_search'] ?? TRUE,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateConfigurationForm(array &$form, FormStateInterface $form_state): void {
    $form_state->setValue(
      'allowed_groups',
      array_values(array_filter($form_state->getValue('allowed_groups')))
    );
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state): void {
    $configuration = $this->plugin->getConfiguration();
    $configuration['allowed_groups'] = $form_state->getValue('allowed_groups');
    $configuration['has_search'] = (bool) $form_state->getValue('has_search');

    $this->plugin->setConfiguration($configuration);
  }

}
