/**
 * @file
 * JavaScript for the Duma audio upload widget with hidden slots and auto-expand.
 */

(function ($, Drupal, once) {
    'use strict';
  
    // Define the new class prefix for consistency with the module name.
    const CLASS_PREFIX = 'duma'; 
  
    /**
     * Main widget behavior
     */
    Drupal.behaviors.dumaWidget = {
      attach: function (context) {
        console.debug(`[DUMA] Attaching DumaWidget behavior to context.`);
  
        // 1. Initialize: ensure hidden rows are properly hidden on page load (for tabledrag support)
        once(`${CLASS_PREFIX}TableRowInit`, `tr.${CLASS_PREFIX}-hidden-row`, context).forEach(function (row) {
          const $row = $(row);
          const $container = $row.find(`.${CLASS_PREFIX}-upload-container`);
          const $targetInput = $container.find(`.${CLASS_PREFIX}-media-id-input`);
          
          if (!$targetInput.val() || $targetInput.val() === '') {
            $row.hide();
          } else {
            $row.removeClass(`${CLASS_PREFIX}-hidden-row`).show();
          }
        });
        
        // 2. Handle file input functionality
        once(`${CLASS_PREFIX}Widget`, `.${CLASS_PREFIX}-upload-container`, context).forEach(function (container) {
          const $container = $(container);
          const $input = $container.find(`.${CLASS_PREFIX}-upload-file-input`);
          const $label = $container.find(`.${CLASS_PREFIX}-upload-label`);
          
          if ($input.length === 0) {
            console.warn('[DUMA] Input file element not found in container.', $container);
            return;
          }
  
          console.debug(`[DUMA] Initializing input for delta: ${$container.attr('data-delta')}`);
          
          // CRITICAL FIX: Ensure Label-Input relationship for native click
          if ($label.length && $input.length) {
            let inputId = $input.attr('id');
            // If the input doesn't have an ID, generate one dynamically
            if (!inputId) {
              inputId = `${CLASS_PREFIX}-file-${$container.attr('data-delta')}-${Date.now()}`;
              $input.attr('id', inputId);
            }
            
            // Set the 'for' attribute on the label to link it to the input ID
            $label.attr('for', inputId);
            
            // Unbind previous handlers to avoid duplicates in AJAX environments
            $label.off('click.duma-delegation');
            
            // Bind handler to stop propagation
            $label.on('click.duma-delegation', function(e) {
              e.stopPropagation();
              console.debug('[DUMA] Label clicked. Relying on native <label for> behavior.');
            });
          }
  
          // Store default label content
          const defaultLabelContent = $label.html();
  
          // --- Bind Change Event (for file selection) ---
          $input.on('change', function (e) {
            if (e.target.files.length === 0) {
              console.debug('[DUMA] No file selected, aborting upload.');
              return;
            }
  
            const file = e.target.files[0];
            const $dataContainer = $container.find(`.${CLASS_PREFIX}-token-url-data`);
            const uploadUrl = $dataContainer.attr('data-upload-url');
            const csrfToken = $dataContainer.attr('data-csrf-token');
            const maxSize = parseInt($dataContainer.attr('data-max-size'), 10);
            
            clearMessages($container);
            
            // Size validation
            if (maxSize > 0 && file.size > maxSize) {
              showError($container, Drupal.t('The audio file is too large. Maximum size is @max_size.', {
                '@max_size': Drupal.formatSize(maxSize)
              }));
              $input.val('');
              return;
            }
  
            // State change: Show loading and disable interaction
            updateUIState($container, 'loading');
            
            console.debug(`[DUMA] Starting upload for file: ${file.name}`);
            
            // Prepare form data with standard field name 'file'
            const formData = new FormData();
            formData.append('file', file);
            
            // AJAX Call with CSRF token in headers
            $.ajax({
              url: uploadUrl,
              type: 'POST',
              data: formData,
              processData: false,
              contentType: false,
              headers: {
                'X-CSRF-Token': csrfToken
              },
              success: function (response) {
                console.debug('[DUMA] Upload success.', response);
                
                // Check for error flag first
                if (response.error) {
                  showError($container, response.message || Drupal.t('Upload failed.'));
                  updateUIState($container, 'default', defaultLabelContent);
                  return;
                }
                
                if (response.media_id) {
                  // Update hidden input with new Media ID
                  const $targetIdInput = $container.find(`.${CLASS_PREFIX}-media-id-input`);
                  $targetIdInput.val(response.media_id);
                  $targetIdInput.trigger('change');
                  
                  // CRITICAL FIX: Find the preview container and update it
                  let $previewContainer = $container.find(`.${CLASS_PREFIX}-upload-preview`);
                  console.debug('[DUMA] Preview container found:', $previewContainer.length);
                  
                  // If not found, create it before the label
                  if ($previewContainer.length === 0) {
                    console.warn('[DUMA] Preview container not found! Creating it dynamically.');
                    const $label = $container.find(`.${CLASS_PREFIX}-upload-label`);
                    if ($label.length) {
                      $label.before('<div class="' + CLASS_PREFIX + '-upload-preview"></div>');
                      $previewContainer = $container.find(`.${CLASS_PREFIX}-upload-preview`);
                      console.debug('[DUMA] Created preview container:', $previewContainer.length);
                    }
                  }
                  
                  // Generate the audio markup
                  const audioMarkup = getAudioMarkup(response.preview_url, response.filename, response.filesize);
                  console.debug('[DUMA] Generated audio markup:', audioMarkup);
                  
                  // Update the preview container
                  if ($previewContainer.length > 0) {
                    $previewContainer.html(audioMarkup);
                    console.debug('[DUMA] Preview HTML after update:', $previewContainer.html());
                  } else {
                    console.error('[DUMA] Could not create or find preview container!');
                  }
                  
                  // Update UI to 'in-use' state and show actions
                  updateUIState($container, 'in-use');
                  showSuccess($container, Drupal.t('Audio file uploaded successfully.'));
  
                  // Multi-upload: check if we need to show the next empty slot
                  autoExpandNextSlot($container);
                } else {
                  showError($container, Drupal.t('Upload failed: No Media ID returned.'));
                  updateUIState($container, 'default', defaultLabelContent);
                }
              },
              error: function (xhr, status, error) {
                console.error(`[DUMA] AJAX Upload Error. Status: ${status}, Error: ${error}`, xhr);
                
                let errorMessage = Drupal.t('An unknown error occurred during upload.');
                
                if (xhr.responseJSON) {
                  if (xhr.responseJSON.message) {
                    errorMessage = xhr.responseJSON.message;
                  } else if (xhr.responseJSON.error && typeof xhr.responseJSON.error === 'string') {
                    errorMessage = xhr.responseJSON.error;
                  }
                } else if (xhr.responseText) {
                  try {
                    const jsonResponse = JSON.parse(xhr.responseText);
                    if (jsonResponse.message) {
                      errorMessage = jsonResponse.message;
                    }
                  } catch (parseError) {
                    if (error) {
                      errorMessage = Drupal.t('Upload failed: @error', {'@error': error});
                    }
                  }
                } else if (error) {
                  errorMessage = Drupal.t('Upload failed: @error', {'@error': error});
                }
                
                showError($container, errorMessage);
                updateUIState($container, 'default', defaultLabelContent);
              }
            });
            
            // Clear the file input value so change event fires again for same file
            $input.val(''); 
          });
          
          // 3. Bind Remove button functionality
          bindRemoveButton($container, defaultLabelContent);
        });
      }
    };
  
    /**
     * Generates the HTML markup for the audio preview (HTML5 <audio> tag).
     */
    function getAudioMarkup(url, filename, filesize) {
      console.debug('[DUMA] getAudioMarkup called with:', {url, filename, filesize});
      
      if (!url) {
        console.warn('[DUMA] No URL provided to getAudioMarkup');
        return '';
      }
  
      // Escape HTML to prevent XSS - but preserve the URL structure
      const safeUrl = url; // URL is already properly encoded from PHP
      const safeFilename = $('<div>').text(filename).html();
      const safeFilesize = filesize ? `(${$('<div>').text(filesize).html()})` : '';
      
      const markup = `<div class="${CLASS_PREFIX}-audio-wrapper">
        <audio controls src="${safeUrl}">
          ${Drupal.t('Your browser does not support the audio element.')}
        </audio>
        <div class="${CLASS_PREFIX}-audio-details">
          <div class="${CLASS_PREFIX}-audio-filename">${safeFilename}</div>
          <div class="${CLASS_PREFIX}-audio-filesize">${safeFilesize}</div>
        </div>
      </div>`;
      
      console.debug('[DUMA] Generated markup:', markup);
      return markup;
    }
  
    /**
     * Updates the UI state of the upload container.
     *
     * @param {jQuery} $container
     * @param {string} state 'default', 'loading', or 'in-use'.
     * @param {string} defaultLabelContent Optional, the original label HTML.
     */
    function updateUIState($container, state, defaultLabelContent) {
      const $label = $container.find(`.${CLASS_PREFIX}-upload-label`);
      const $actions = $container.find(`.${CLASS_PREFIX}-upload-actions`);
      
      console.debug(`[DUMA] Updating UI state to: ${state}`);
      
      // Reset all state classes
      $container.removeClass(`${CLASS_PREFIX}-slot-default ${CLASS_PREFIX}-slot-loading ${CLASS_PREFIX}-slot-in-use`);
        
      // Apply new state classes
      if (state === 'default') {
        $container.addClass(`${CLASS_PREFIX}-slot-default`);
        $actions.addClass(`${CLASS_PREFIX}-hidden-actions`);
        if (defaultLabelContent) {
          $label.html(defaultLabelContent);
        }
      } else if (state === 'loading') {
        $container.addClass(`${CLASS_PREFIX}-slot-loading`);
        $label.html(Drupal.t('Uploading... Please wait.'));
      } else if (state === 'in-use') {
        $container.addClass(`${CLASS_PREFIX}-slot-in-use`);
        $actions.removeClass(`${CLASS_PREFIX}-hidden-actions`);
        $label.text(Drupal.t('Click to replace audio file'));
      }
    }
  
    /**
     * For multi-value fields, shows the next empty slot.
     */
    function autoExpandNextSlot($currentContainer) {
      if (!$currentContainer.hasClass(`${CLASS_PREFIX}-multi-enabled`)) {
        return;
      }
      
      const $nextContainer = $currentContainer.parent().next().find(`.${CLASS_PREFIX}-upload-container`);
      
      if ($nextContainer.length && $nextContainer.hasClass(`${CLASS_PREFIX}-hidden-slot`)) {
        console.debug('[DUMA] Auto-expanding next slot.');
        $nextContainer.removeClass(`${CLASS_PREFIX}-hidden-slot`);
        
        const $nextTableRow = $nextContainer.closest('tr');
        if ($nextTableRow.length) {
          $nextTableRow.removeClass(`${CLASS_PREFIX}-hidden-row`).show();
        }
      }
    }
  
    /**
     * Binds the click handler for the 'Remove' button.
     */
    function bindRemoveButton($container, defaultLabelContent) {
      $container.find(`.${CLASS_PREFIX}-remove-button`).on('click', function (e) {
        e.preventDefault();
        
        const $input = $container.find(`.${CLASS_PREFIX}-upload-file-input`);
        const $targetIdInput = $container.find(`.${CLASS_PREFIX}-media-id-input`);
        
        console.debug('[DUMA] Remove button clicked. Clearing fields.');
  
        $targetIdInput.val('');
        $input.val('');
        $targetIdInput.trigger('change');
  
        updateUIState($container, 'default', defaultLabelContent);
        clearMessages($container);
        
        $container.find(`.${CLASS_PREFIX}-upload-preview`).empty();
        
        if (!$container.hasClass(`${CLASS_PREFIX}-delta-0`)) {
          console.debug('[DUMA] Hiding removed non-first slot.');
          $container.addClass(`${CLASS_PREFIX}-hidden-slot`).removeClass(`${CLASS_PREFIX}-slot-in-use`);
          
          const $tableRow = $container.closest('tr');
          if ($tableRow.length) {
            $tableRow.addClass(`${CLASS_PREFIX}-hidden-row`).hide();
          }
        }
        
        return false;
      });
    }
  
    /**
     * Helper functions for messages
     */
    function showError($container, message) {
      const $errorContainer = $container.find(`.${CLASS_PREFIX}-error-message`);
      $errorContainer.text(message).addClass('error').removeClass('success');
    }
  
    function showSuccess($container, message) {
      const $errorContainer = $container.find(`.${CLASS_PREFIX}-error-message`);
      $errorContainer.text(message).addClass('success').removeClass('error');
    }
  
    function clearMessages($container) {
      const $errorContainer = $container.find(`.${CLASS_PREFIX}-error-message`);
      $errorContainer.text('').removeClass('error success');
    }
  
  })(jQuery, Drupal, once);