<?php

namespace Drupal\duma\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\File\FileSystemInterface;
use Drupal\file\FileRepositoryInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\JsonResponse;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\TempStore\PrivateTempStoreFactory;

/**
 * Controller for DUMA file upload functionality.
 */
class DumaController extends ControllerBase {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountInterface
   */
  protected $currentUser;

  /**
   * The file system service.
   *
   * @var \Drupal\Core\File\FileSystemInterface
   */
  protected $fileSystem;

  /**
   * The file repository service.
   *
   * @var \Drupal\file\FileRepositoryInterface
   */
  protected $fileRepository;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The logger factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected $loggerFactory;

  /**
   * The private tempstore factory.
   *
   * @var \Drupal\Core\TempStore\PrivateTempStoreFactory
   */
  protected $tempStoreFactory;

  /**
   * Constructs a new DumaController object.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Session\AccountInterface $current_user
   *   The current user.
   * @param \Drupal\Core\File\FileSystemInterface $file_system
   *   The file system service.
   * @param \Drupal\file\FileRepositoryInterface $file_repository
   *   The file repository service.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory.
   * @param \Drupal\Core\TempStore\PrivateTempStoreFactory $temp_store_factory
   *   The private tempstore factory.
   */
  public function __construct(
    EntityTypeManagerInterface $entity_type_manager,
    AccountInterface $current_user,
    FileSystemInterface $file_system,
    FileRepositoryInterface $file_repository,
    ConfigFactoryInterface $config_factory,
    LoggerChannelFactoryInterface $logger_factory,
    PrivateTempStoreFactory $temp_store_factory
  ) {
    $this->entityTypeManager = $entity_type_manager;
    $this->currentUser = $current_user;
    $this->fileSystem = $file_system;
    $this->fileRepository = $file_repository;
    $this->configFactory = $config_factory;
    $this->loggerFactory = $logger_factory;
    $this->tempStoreFactory = $temp_store_factory;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('entity_type.manager'),
      $container->get('current_user'),
      $container->get('file_system'),
      $container->get('file.repository'),
      $container->get('config.factory'),
      $container->get('logger.factory'),
      $container->get('tempstore.private')
    );
  }

  /**
   * Handles file upload from the DUMA widget.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request object.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   A JSON response with the upload result.
   */
  public function uploadFile(Request $request) {
    try {
      // Check rate limiting
      if (!$this->checkRateLimit()) {
        return new JsonResponse([
          'error' => $this->t('Upload rate limit exceeded. Please try again later.'),
        ], 429);
      }

      // Get the uploaded file (JavaScript sends as 'file', not 'files')
      $uploaded_file = $request->files->get('file');
      if (!$uploaded_file) {
        return new JsonResponse([
          'error' => $this->t('No file was uploaded.'),
        ], 400);
      }
      
      // Get configuration
      $config = $this->configFactory->get('duma.settings');
      $media_type = $config->get('media_type') ?? 'audio';
      $source_field = $config->get('source_field') ?? 'field_media_audio_file';
      $max_size = (int) ($config->get('max_size') ?? 52428800);

      // Validate file size
      if ($max_size > 0 && $uploaded_file->getSize() > $max_size) {
        $max_size_mb = round($max_size / 1048576, 2);
        return new JsonResponse([
          'error' => $this->t('File size exceeds the maximum allowed size of @size MB.', [
            '@size' => $max_size_mb,
          ]),
        ], 400);
      }

      // Validate file type (audio files only)
      $mime_type = $uploaded_file->getMimeType();
      $original_filename = $uploaded_file->getClientOriginalName();
      $extension = strtolower(pathinfo($original_filename, PATHINFO_EXTENSION));
      
      $allowed_mime_types = [
        'audio/mpeg',        // .mp3
        'audio/mp3',
        'audio/wav',         // .wav
        'audio/wave',
        'audio/x-wav',
        'audio/ogg',         // .ogg, .opus
        'audio/opus',
        'application/ogg',   // .ogg, .opus (alternative)
        'audio/mp4',         // .m4a
        'audio/x-m4a',
        'audio/flac',        // .flac
        'audio/x-flac',
        'audio/aac',         // .aac
        'audio/aacp',
        'audio/webm',        // .webm
        'application/octet-stream', // Generic binary - we'll check extension
      ];
      
      $allowed_extensions = [
        'mp3', 'wav', 'ogg', 'opus', 'm4a', 'flac', 'aac', 'webm',
      ];
      
      // Check MIME type OR extension (since MIME detection can be unreliable)
      $mime_valid = in_array($mime_type, $allowed_mime_types);
      $extension_valid = in_array($extension, $allowed_extensions);
      
      if (!$mime_valid && !$extension_valid) {
        $this->loggerFactory->get('duma')->warning('Rejected file with MIME type: @mime and extension: @ext', [
          '@mime' => $mime_type,
          '@ext' => $extension,
        ]);
        return new JsonResponse([
          'error' => $this->t('File type not allowed. Detected type: @type, extension: @ext. Only audio files are allowed.', [
            '@type' => $mime_type,
            '@ext' => $extension,
          ]),
        ], 400);
      }
      
      // Log when using octet-stream fallback
      if ($mime_type === 'application/octet-stream') {
        $this->loggerFactory->get('duma')->info('Accepting file with generic MIME type based on extension: @ext', [
          '@ext' => $extension,
        ]);
      }

      // Prepare destination directory
      $destination_dir = 'public://duma_uploads';
      $this->fileSystem->prepareDirectory($destination_dir, FileSystemInterface::CREATE_DIRECTORY | FileSystemInterface::MODIFY_PERMISSIONS);

      // Save the file
      $file = $this->fileRepository->writeData(
        file_get_contents($uploaded_file->getPathname()),
        $destination_dir . '/' . $uploaded_file->getClientOriginalName(),
        FileSystemInterface::EXISTS_RENAME
      );

      if (!$file) {
        throw new \Exception('Failed to save file.');
      }

      // Make file permanent
      $file->setPermanent();
      $file->save();

      // Create media entity
      $media_storage = $this->entityTypeManager->getStorage('media');
      $media = $media_storage->create([
        'bundle' => $media_type,
        'name' => $uploaded_file->getClientOriginalName(),
        $source_field => [
          'target_id' => $file->id(),
        ],
        'uid' => $this->currentUser->id(),
      ]);
      $media->save();

      $this->loggerFactory->get('duma')->info('Audio file uploaded: @filename (Media ID: @mid)', [
        '@filename' => $uploaded_file->getClientOriginalName(),
        '@mid' => $media->id(),
      ]);

      // Get file size for display
      $filesize = $file->getSize();
      $filesize_markup = \Drupal\Core\StringTranslation\ByteSizeMarkup::create($filesize);

      return new JsonResponse([
        'success' => TRUE,
        'media_id' => $media->id(),
        'file_id' => $file->id(),
        'filename' => $file->getFilename(),
        'preview_url' => $file->createFileUrl(),
        'filesize' => (string) $filesize_markup,
      ]);
    }
    catch (\Exception $e) {
      $this->loggerFactory->get('duma')->error('Error uploading file: @message', [
        '@message' => $e->getMessage(),
      ]);

      return new JsonResponse([
        'error' => $this->t('An error occurred while uploading the file. Please try again.'),
      ], 500);
    }
  }

  /**
   * Checks if the current user has exceeded the upload rate limit.
   *
   * @return bool
   *   TRUE if upload is allowed, FALSE if rate limit exceeded.
   */
  protected function checkRateLimit() {
    $config = $this->configFactory->get('duma.settings');
    $max_uploads = (int) ($config->get('max_uploads_per_minute') ?? 60);

    // Get tempstore for this user
    $tempstore = $this->tempStoreFactory->get('duma');
    $upload_times = $tempstore->get('upload_times') ?? [];

    // Remove uploads older than 1 minute
    $current_time = time();
    $upload_times = array_filter($upload_times, function($timestamp) use ($current_time) {
      return ($current_time - $timestamp) < 60;
    });

    // Check if limit exceeded
    if (count($upload_times) >= $max_uploads) {
      return FALSE;
    }

    // Add current upload time
    $upload_times[] = $current_time;
    $tempstore->set('upload_times', $upload_times);

    return TRUE;
  }

}