<?php

namespace Drupal\duma\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\File\FileUrlGeneratorInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Config\TypedConfigManagerInterface;

/**
 * Configure DUMA settings for this site.
 */
class DumaSettingsForm extends ConfigFormBase {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The file URL generator.
   *
   * @var \Drupal\Core\File\FileUrlGeneratorInterface
   */
  protected $fileUrlGenerator;

  /**
   * The logger factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected $loggerFactory;

  /**
   * Constructs a new DumaSettingsForm object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The factory for configuration objects.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\File\FileUrlGeneratorInterface $file_url_generator
   *   The file URL generator.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory.
   * @param \Drupal\Core\Config\TypedConfigManagerInterface $typed_config
   *   The typed configuration manager.
   */
  public function __construct(ConfigFactoryInterface $config_factory, EntityTypeManagerInterface $entity_type_manager, FileUrlGeneratorInterface $file_url_generator, LoggerChannelFactoryInterface $logger_factory, TypedConfigManagerInterface $typed_config) {
    parent::__construct($config_factory, $typed_config);
    $this->entityTypeManager = $entity_type_manager;
    $this->fileUrlGenerator = $file_url_generator;
    $this->loggerFactory = $logger_factory;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('entity_type.manager'),
      $container->get('file_url_generator'),
      $container->get('logger.factory'),
      $container->get('config.typed')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'duma_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['duma.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('duma.settings');

    // General Settings section
    $form['general'] = [
      '#type' => 'details',
      '#title' => $this->t('General Settings'),
      '#open' => TRUE,
    ];

    $form['general']['media_type'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Media Type'),
      '#description' => $this->t('The machine name of the media type to use for audio uploads (e.g., "audio").'),
      '#default_value' => $config->get('media_type') ?? 'audio',
      '#required' => TRUE,
    ];

    $form['general']['source_field'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Source Field'),
      '#description' => $this->t('The machine name of the file field on the media type (e.g., "field_media_audio_file").'),
      '#default_value' => $config->get('source_field') ?? 'field_media_audio_file',
      '#required' => TRUE,
    ];

    // File Upload Settings section
    $form['upload_settings'] = [
      '#type' => 'details',
      '#title' => $this->t('File Upload Settings'),
      '#open' => TRUE,
    ];

    // Convert bytes to MB for display
    $max_size_bytes = $config->get('max_size') ?? 52428800;
    $max_size_mb = round($max_size_bytes / 1048576, 2);

    $form['upload_settings']['max_size'] = [
      '#type' => 'number',
      '#title' => $this->t('Maximum File Size (in MB)'),
      '#description' => $this->t('Maximum file size in megabytes. Default is 50 MB. Set to 0 for no limit.'),
      '#default_value' => $max_size_mb,
      '#min' => 0,
      '#max' => 500,
      '#step' => 0.01,
      '#required' => TRUE,
    ];

    $form['upload_settings']['max_uploads_per_minute'] = [
      '#type' => 'number',
      '#title' => $this->t('Maximum Uploads Per Minute'),
      '#description' => $this->t('Maximum number of file uploads allowed per user per minute. Default is 60.'),
      '#default_value' => $config->get('max_uploads_per_minute') ?? 60,
      '#min' => 1,
      '#max' => 1000,
      '#required' => TRUE,
    ];

    // Get DUMA fields for both tables and custom labels
    $duma_fields = $this->findDumaFields();

    // Fields table section
    $form['fields_table'] = [
      '#type' => 'details',
      '#title' => $this->t('Fields using the DUMA widget'),
      '#description' => $this->t('This table shows all entity reference fields currently configured to use the DUMA audio recorder widget.'),
      '#open' => TRUE,
    ];

    if (empty($duma_fields)) {
      $form['fields_table']['no_fields'] = [
        '#markup' => '<p>' . $this->t('No fields are currently configured to use the DUMA widget.') . '</p>',
      ];
    } else {
      $header = [
        'entity_type' => $this->t('Entity Type'),
        'bundle' => $this->t('Bundle'),
        'field_name' => $this->t('Field Name'),
        'field_label' => $this->t('Field Label'),
        'form_mode' => $this->t('Form Mode'),
        'label_image' => $this->t('Custom Label Image'),
      ];

      $rows = [];
      foreach ($duma_fields as $field_info) {
        $bundle = $field_info['bundle'];
        $custom_label_fid = $config->get('labels.' . $bundle);
        $custom_label_image = $this->t('No image');
        
        try {
          if ($custom_label_fid && $file = $this->entityTypeManager->getStorage('file')->load($custom_label_fid)) {
            $file_uri = $file->getFileUri();
            $style = 'style="max-width: 100px; height: auto;"';
            $custom_label_image = '<img src="' . $this->fileUrlGenerator->generate($file_uri)->toString() . '" alt="' . $file->getFilename() . '" ' . $style . ' />';
          }
        }
        catch (\Exception $e) {
          $custom_label_image = $this->t('Error loading image');
        }

        $rows[] = [
          'entity_type' => $field_info['entity_type'],
          'bundle' => $field_info['bundle_label'],
          'field_name' => $field_info['field_name'],
          'field_label' => $field_info['field_label'],
          'form_mode' => $field_info['form_mode'] === 'default' ? $this->t('Default') : $field_info['form_mode'],
          'label_image' => [
            'data' => [
              '#markup' => $custom_label_image,
            ],
          ],
        ];
      }

      $form['fields_table']['table'] = [
        '#type' => 'table',
        '#header' => $header,
        '#rows' => $rows,
        '#empty' => $this->t('No fields found.'),
      ];

      $form['fields_table']['count'] = [
        '#markup' => '<p><strong>' . $this->t('Total: @count field(s)', ['@count' => count($duma_fields)]) . '</strong></p>',
      ];
    }

    // Custom Widget Labels section
    $form['custom_labels'] = [
      '#type' => 'details',
      '#title' => $this->t('Custom Widget Labels'),
      '#open' => TRUE,
    ];
    $this->addCustomLabelSection($form['custom_labels'], $duma_fields);

    return parent::buildForm($form, $form_state);
  }

  /**
   * Adds the custom label upload section to the form.
   *
   * @param array $form
   *   The form array.
   * @param array $duma_fields
   *   An array of fields using the DUMA widget.
   */
  protected function addCustomLabelSection(array &$form, array $duma_fields) {
    $config = $this->config('duma.settings');
    $bundles = array_unique(array_column($duma_fields, 'bundle'));

    if (empty($bundles)) {
      $form['info'] = [
        '#markup' => $this->t('No DUMA fields are currently in use on any content type.'),
      ];
      return;
    }

    $form['info'] = [
      '#markup' => $this->t('Upload a custom label image for each content type. Leave blank to use the default text.'),
    ];

    // Get max file size from config for validators
    $max_size_bytes = $config->get('max_size') ?? 52428800;
    $max_size_mb = round($max_size_bytes / 1048576, 2);

    foreach ($bundles as $bundle) {
      // Get bundle label
      $bundle_label = $this->getBundleLabel($duma_fields, $bundle);
      
      $fid = $config->get('labels.' . $bundle);

      $form[$bundle] = [
        '#type' => 'managed_file',
        '#title' => $this->t('Label for %bundle', ['%bundle' => $bundle_label]),
        '#upload_location' => 'public://duma_labels',
        '#default_value' => $fid ? [$fid] : [],
        '#upload_validators' => [
          'FileExtension' => [
            'extensions' => 'jpg jpeg png gif svg',
          ],
          'FileSizeLimit' => [
            'fileLimit' => $max_size_bytes,
          ],
        ],
        '#description' => $this->t('Upload an image to use as a custom label for the DUMA widget on this content type. Recommended formats: PNG, SVG. Maximum size: @size MB.', [
          '@size' => $max_size_mb,
        ]),
      ];
    }
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    parent::validateForm($form, $form_state);
    
    $max_size = $form_state->getValue('max_size');
    if (!is_numeric($max_size) || $max_size < 0) {
      $form_state->setErrorByName('max_size', $this->t('Maximum file size must be a positive number.'));
    }
    
    if ($max_size > 500) {
      $form_state->setErrorByName('max_size', $this->t('Maximum file size cannot exceed 500 MB.'));
    }

    $max_uploads = $form_state->getValue('max_uploads_per_minute');
    if ($max_uploads < 1 || $max_uploads > 1000) {
      $form_state->setErrorByName('max_uploads_per_minute', $this->t('Maximum uploads per minute must be between 1 and 1000.'));
    }

    // Validate media type exists
    $media_type = $form_state->getValue('media_type');
    try {
      if ($this->entityTypeManager->hasDefinition('media_type')) {
        $media_type_entity = $this->entityTypeManager->getStorage('media_type')->load($media_type);
        if (!$media_type_entity) {
          $form_state->setErrorByName('media_type', $this->t('The media type "@type" does not exist.', ['@type' => $media_type]));
        }
      }
    }
    catch (\Exception $e) {
      $this->loggerFactory->get('duma')->error('Error validating media type: @message', [
        '@message' => $e->getMessage(),
      ]);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    // Convert MB to bytes for storage
    $max_size_mb = $form_state->getValue('max_size');
    $max_size_bytes = $max_size_mb * 1048576;

    try {
      $config = $this->config('duma.settings');
      $config
        ->set('media_type', $form_state->getValue('media_type'))
        ->set('source_field', $form_state->getValue('source_field'))
        ->set('max_size', (string) $max_size_bytes)
        ->set('max_uploads_per_minute', $form_state->getValue('max_uploads_per_minute'));

      // Handle custom label uploads
      $duma_fields = $this->findDumaFields();
      $bundles = array_unique(array_column($duma_fields, 'bundle'));
      
      foreach ($bundles as $bundle) {
        $fid_array = $form_state->getValue($bundle);
        $fid = NULL;
        
        // Handle both array and single value formats
        if (is_array($fid_array) && !empty($fid_array)) {
          $fid = reset($fid_array);
        } elseif (!is_array($fid_array) && !empty($fid_array)) {
          $fid = $fid_array;
        }
        
        $config->set('labels.' . $bundle, $fid);
        
        if ($fid) {
          try {
            $file = $this->entityTypeManager->getStorage('file')->load($fid);
            if ($file) {
              // Make the file permanent and add usage
              $file->setPermanent();
              $file->save();
              \Drupal::service('file.usage')->add($file, 'duma', 'duma_settings', $this->getFormId());
            }
          }
          catch (\Exception $e) {
            $this->loggerFactory->get('duma')->error('Failed to add file usage for FID @fid: @message', [
              '@fid' => $fid,
              '@message' => $e->getMessage(),
            ]);
          }
        }
      }

      $config->save();

      // Clear config cache to ensure changes take effect
      \Drupal::service('cache.config')->deleteAll();

      $this->loggerFactory->get('duma')->info('DUMA settings updated: max_size=@size MB, max_uploads=@uploads', [
        '@size' => $max_size_mb,
        '@uploads' => $form_state->getValue('max_uploads_per_minute'),
      ]);
    }
    catch (\Exception $e) {
      $this->loggerFactory->get('duma')->error('Error saving DUMA settings: @message', [
        '@message' => $e->getMessage(),
      ]);
      $this->messenger()->addError($this->t('An error occurred while saving settings. Please check the logs.'));
      return;
    }

    parent::submitForm($form, $form_state);
  }

  /**
   * Finds all fields that are using the DUMA widget.
   *
   * @return array
   *   An array of field information including entity type, bundle, field name, and form mode.
   */
  protected function findDumaFields() {
    $duma_fields = [];
    
    try {
      $form_displays = $this->entityTypeManager->getStorage('entity_form_display')->loadMultiple();

      foreach ($form_displays as $display) {
        $components = $display->getComponents();
        foreach ($components as $field_name => $component) {
          if (isset($component['type']) && $component['type'] === 'duma_widget') {
            $entity_type_id = $display->getTargetEntityTypeId();
            $bundle = $display->getTargetBundle();
            $field_config_id = $entity_type_id . '.' . $bundle . '.' . $field_name;
            
            // Check if field config exists before trying to load it
            $field_config = $this->entityTypeManager->getStorage('field_config')->load($field_config_id);
            if ($field_config) {
              // Get bundle label
              $bundle_label = $this->getBundleLabelFromEntityType($entity_type_id, $bundle);
              
              $duma_fields[] = [
                'entity_type' => $entity_type_id,
                'bundle' => $bundle,
                'bundle_label' => $bundle_label,
                'field_name' => $field_name,
                'field_label' => $field_config->label(),
                'form_mode' => $display->getMode(),
              ];
            }
          }
        }
      }
    }
    catch (\Exception $e) {
      $this->loggerFactory->get('duma')->error('Error finding DUMA fields: @message', [
        '@message' => $e->getMessage(),
      ]);
    }

    return $duma_fields;
  }

  /**
   * Gets the human-readable label for a bundle from the fields array.
   *
   * @param array $duma_fields
   *   The array of DUMA fields.
   * @param string $bundle
   *   The bundle machine name.
   *
   * @return string
   *   The bundle label.
   */
  protected function getBundleLabel(array $duma_fields, $bundle) {
    foreach ($duma_fields as $field_info) {
      if ($field_info['bundle'] === $bundle) {
        return $field_info['bundle_label'];
      }
    }
    return $bundle;
  }

  /**
   * Gets the human-readable label for a bundle from entity type.
   *
   * @param string $entity_type_id
   *   The entity type ID.
   * @param string $bundle
   *   The bundle machine name.
   *
   * @return string
   *   The bundle label or the machine name if not found.
   */
  protected function getBundleLabelFromEntityType($entity_type_id, $bundle) {
    $bundle_label = $bundle;
    
    try {
      // Map entity types to their bundle entity types
      $bundle_entity_types = [
        'node' => 'node_type',
        'taxonomy_term' => 'taxonomy_vocabulary',
        'media' => 'media_type',
        'block_content' => 'block_content_type',
        'commerce_product' => 'commerce_product_type',
        'paragraph' => 'paragraphs_type',
      ];
      
      if (isset($bundle_entity_types[$entity_type_id])) {
        $bundle_entity_type = $bundle_entity_types[$entity_type_id];
        
        if ($this->entityTypeManager->hasDefinition($bundle_entity_type)) {
          $bundle_entity = $this->entityTypeManager->getStorage($bundle_entity_type)->load($bundle);
          if ($bundle_entity) {
            $bundle_label = $bundle_entity->label();
          }
        }
      }
    }
    catch (\Exception $e) {
      $this->loggerFactory->get('duma')->warning('Could not load bundle label for @entity_type.@bundle: @message', [
        '@entity_type' => $entity_type_id,
        '@bundle' => $bundle,
        '@message' => $e->getMessage(),
      ]);
    }
    
    return $bundle_label;
  }

}